/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.auditlog;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsStructuredType;
import com.sap.cds.services.utils.DraftUtils;
import com.sap.cds.services.utils.model.CdsAnnotations;
import com.sap.cds.util.CdsModelUtils;

/**
 * This class provides additional metadata about a {@link CdsStructuredType} which is required for Personal Data analysis. Instances of this
 * class can be put into a cache to avoid evaluating the metadata multiple times.
 */
class PersonalDataMeta {
	private static final Logger logger = LoggerFactory.getLogger(PersonalDataMeta.class);

	private static final String FIELD_SEMANTICS_DATASUBJECTID = "DataSubjectID";
	private static final String ENTITY_SEMANTICS_DATASUBJECT = "DataSubject";
	private static final String ENTITY_SEMANTICS_DATASUBJECTDETAILS = "DataSubjectDetails";
	private static final String ENTITY_SEMANTICS_OTHER = "Other";

	private final CdsStructuredType type;
	private final boolean hasPersonalData;
	private final boolean dataSubject;
	private final boolean dataSubjectDetails;
	private final boolean other;
	private final String dataSubjectRole;
	private Set<String> personalDataNames;
	private Set<String> sensitiveDataNames;
	private List<String> keyNames;
	private List<CdsElement> dataSubjectIds;
	private CdsElement dataSubjectAssociation;

	/**
	 * Constructs a new {@link PersonalDataMeta} instance for a given structured type.
	 *
	 * @param type the structured type of the element
	 */
	PersonalDataMeta(CdsStructuredType type) {
		this.type = Objects.requireNonNull(type, "type must not be null");
		boolean isDraftShadowEntity = DraftUtils.isDraftEnabled(type) && type.getQualifiedName().endsWith("_drafts");

		// evaluate @PersonalData.EntitySemantics
		String entitySemantics = getEntitySemantics(this.type);
		this.hasPersonalData = entitySemantics != null && !isDraftShadowEntity;
		this.dataSubject = this.hasPersonalData && ENTITY_SEMANTICS_DATASUBJECT.equals(entitySemantics);
		this.dataSubjectDetails = this.hasPersonalData && ENTITY_SEMANTICS_DATASUBJECTDETAILS.equals(entitySemantics);
		this.other = this.hasPersonalData && ENTITY_SEMANTICS_OTHER.equals(entitySemantics);

		// evaluate @PersonalData.DataSubjectRole
		this.dataSubjectRole = CdsAnnotations.PERSONALDATA_DATASUBJECTROLE.getOrDefault(type);
	}

	/**
	 * The unique key of this instance.
	 *
	 * @return the unique key of this instance.
	 */
	String getKey() {
		return this.type.getQualifiedName();
	}

	/**
	 * Returns a role, if this instance represents a data subject, otherwise an empty string.
	 *
	 * @return a role, if this instance represents a data subject, otherwise an empty string
	 */
	String getDataSubjectRole() {
		return this.dataSubjectRole;
	}

	/**
	 * Indicates whether the {@link CdsStructuredType entity} could contain personal data or not.
	 *
	 * @return <code>true</code> if entity could contain personal data, otherwise <code>false</code>
	 */
	boolean hasPersonalData() {
		return this.hasPersonalData;
	}

	/**
	 * Returns a {@link List list} of {@link CdsElement fields} that are annotated with
	 * {@code @PersonalData.FieldSemantics : 'DataSubjectID'}.
	 *
	 * @return a {@link List list} of {@link CdsElement fields}
	 */
	List<CdsElement> getDataSubjectIds() {
		if (this.dataSubjectIds == null) {
			this.dataSubjectIds = Collections.unmodifiableList(this.type.elements().filter(element -> {
				String annotationValue = CdsAnnotations.PERSONALDATA_FIELDSEMANTICS.getOrDefault(element);
				return FIELD_SEMANTICS_DATASUBJECTID.equals(annotationValue);
			}).collect(Collectors.toList()));
		}
		return this.dataSubjectIds;
	}

	/**
	 * Returns a {@link List} with key names of current {@link CdsStructuredType entity}.
	 *
	 * @return a {@link List} with key names
	 */
	List<String> getKeyNames() {
		if (this.keyNames == null) {
			this.keyNames = Collections.unmodifiableList(new ArrayList<>(CdsModelUtils.keyNames(this.type)));
		}
		return this.keyNames;
	}

	/**
	 * Returns a {@link Set set} of element names which are annotated with {@link CdsAnnotations#PERSONALDATA_POTENTIALLYSENSITIVE}.
	 *
	 * @return a {@link Set set} with element names, could be empty but not null.
	 */
	Set<String> getSensitiveDataNames() {
		if (this.sensitiveDataNames == null) {
			this.sensitiveDataNames = Collections.unmodifiableSet(this.type.elements()
					.filter(CdsAnnotations.PERSONALDATA_POTENTIALLYSENSITIVE::isTrue).map(CdsElement::getName).collect(Collectors.toSet()));
		}
		return this.sensitiveDataNames;
	}

	/**
	 * Returns a {@link Set set} of element names which are annotated with {@link CdsAnnotations#PERSONALDATA_POTENTIALLYPERSONAL} or
	 * {@link CdsAnnotations#PERSONALDATA_POTENTIALLYSENSITIVE}.
	 *
	 * @return a {@link Set set} with element names, could be empty but not null.
	 */
	Set<String> getPersonalDataNames() {
		if (this.personalDataNames == null) {
			this.personalDataNames = Collections.unmodifiableSet(this.type.elements()
					.filter(element -> CdsAnnotations.PERSONALDATA_POTENTIALLYPERSONAL.isTrue(element)
							|| CdsAnnotations.PERSONALDATA_POTENTIALLYSENSITIVE.isTrue(element))
					.map(CdsElement::getName).collect(Collectors.toSet()));
		}
		return this.personalDataNames;
	}

	/**
	 * Indicates whether the entity is a data subject or not.
	 *
	 * @return <code>true</code> if entity is the data subject
	 */
	boolean isDataSubject() {
		return this.dataSubject;
	}

	/**
	 * Indicates whether the entity contains data subject details or not.
	 *
	 * @return <code>true</code> if entity contains data subject details
	 */
	boolean isDataSubjectDetails() {
		return this.dataSubjectDetails;
	}

	/**
	 * Indicates whether the entity contains personal data with entity semantics "Other".
	 *
	 * @return <code>true</code> if entity contains personal data with entity semantics "Other"
	 */
	boolean isOther() {
		return this.other;
	}

	/**
	 * Returns the {@link CdsElement element} that is an association and annotated with
	 * <code>@PersonalData.FieldSemantics : 'DataSubjectID'</code>.
	 *
	 * @return an association to the data subject or <code>null</code>
	 */
	CdsElement getDataSubjectAssociation() {
		if (this.dataSubjectAssociation == null) {
			// TODO: how to handle multiple associations to data subjects
			this.dataSubjectAssociation = getDataSubjectIds().stream() //
					.filter(element -> element.getType().isAssociation()) //
					.findFirst().orElse(null);
		}
		return this.dataSubjectAssociation;
	}

	/**
	 * Returns a {@link Map} containing all entity keys including their values.
	 *
	 * @param row the data providing the values
	 * @return a {@link Map} containing all entity keys including their values
	 */
	Map<String, Object> getKeys(Map<String, Object> row) {
		Map<String, Object> matcher = new HashMap<>();
		getKeyNames().stream().filter(row::containsKey).forEach(keyField -> matcher.put(keyField, row.get(keyField)));
		return matcher;
	}

	@Override
	public String toString() {
		return getKey();
	}

	// static helpers

	/**
	 * Returns the value of the annotation {@code @PersonalData.EntitySemantics} or <code>null</code> if not annotated.
	 *
	 * @param entity the entity whose annotation {@code @PersonalData.EntitySemantics} will be read
	 * @return the entity semantics or <code>null</code>
	 */
	private static String getEntitySemantics(CdsStructuredType entity) {
		String entitySemantics = CdsAnnotations.PERSONALDATA_ENTITYSEMANTICS.getOrDefault(entity);
		if (entitySemantics != null) {
			switch (entitySemantics) {
			case ENTITY_SEMANTICS_DATASUBJECT:
			case ENTITY_SEMANTICS_DATASUBJECTDETAILS:
			case ENTITY_SEMANTICS_OTHER:
				return entitySemantics;
			default:
				logger.warn("Entity semantics '{}' not supported.", entitySemantics);
				break;
			}
		}
		return null;
	}
}