/*********************************************************************
 * (C) 2024 SAP SE or an SAP affiliate company. All rights reserved. *
 *********************************************************************/
package com.sap.cds.services.impl.authorization;

import static com.sap.cds.services.impl.authorization.ReferenceAssociationTraverser.NO_COMPOSITIONS;
import static com.sap.cds.services.impl.authorization.ReferenceAssociationTraverser.traverseAsAbsolute;
import static com.sap.cds.services.impl.authorization.ReferenceAssociationTraverser.traverseAsRelative;
import static com.sap.cds.util.CqnStatementUtils.targetEntity;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

import com.sap.cds.impl.builder.model.ExpandBuilder;
import com.sap.cds.impl.builder.model.StructuredTypeRefImpl;
import com.sap.cds.ql.CQL;
import com.sap.cds.ql.Predicate;
import com.sap.cds.ql.Select;
import com.sap.cds.ql.cqn.CqnElementRef;
import com.sap.cds.ql.cqn.CqnExistsSubquery;
import com.sap.cds.ql.cqn.CqnExpand;
import com.sap.cds.ql.cqn.CqnInline;
import com.sap.cds.ql.cqn.CqnMatchPredicate;
import com.sap.cds.ql.cqn.CqnPredicate;
import com.sap.cds.ql.cqn.CqnReference;
import com.sap.cds.ql.cqn.CqnReference.Segment;
import com.sap.cds.ql.cqn.CqnSelectListItem;
import com.sap.cds.ql.cqn.CqnStructuredTypeRef;
import com.sap.cds.ql.cqn.CqnValue;
import com.sap.cds.ql.cqn.Modifier;
import com.sap.cds.ql.impl.ExpressionVisitor;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsModel;
import com.sap.cds.reflect.CdsStructuredType;
import com.sap.cds.reflect.impl.reader.model.CdsConstants;
import com.sap.cds.services.authorization.AuthorizationService;
import com.sap.cds.services.cds.CdsReadEventContext;
import com.sap.cds.services.draft.Drafts;
import com.sap.cds.services.utils.DraftUtils;
import com.sap.cds.util.CdsModelUtils;

@SuppressWarnings("UnstableApiUsage")
public class ReadStatementAuthorizationModifier implements Modifier {

	private static final Predicate IS_ACTIVE_EQ_FALSE = CQL.get(Drafts.IS_ACTIVE_ENTITY).eq(false);
	private final AuthorizationService authorizationService;
	private final CdsModel model;
	private final String event;
	private final CdsStructuredType parentTarget; // target entity of a parent statement -> e.g. query in the source, needed only for $outer
	private final CdsStructuredType target; // target of the current statement

	public ReadStatementAuthorizationModifier(AuthorizationService authService, CdsReadEventContext context) {
		this(authService, context.getModel(), context.getEvent(),
			null, // No parent on root call
			targetEntity(context.getCqn(), context.getModel()));
	}

	private ReadStatementAuthorizationModifier(AuthorizationService authorizationService, CdsModel model, String event,
			CdsStructuredType parentTarget, CdsStructuredType target) {
		this.authorizationService = Objects.requireNonNull(authorizationService);
		this.model = Objects.requireNonNull(model);
		this.event = event;
		this.target = target;
		this.parentTarget = parentTarget;
	}

	@Override
	public CqnStructuredTypeRef ref(CqnStructuredTypeRef ref) {
		Segment targetSegment = ref.targetSegment();

		List<Segment> segments = new ArrayList<>(ref.segments());
		CdsEntity root = model.getEntity(ref.firstSegment());
		traverseAsAbsolute(root, segments, NO_COMPOSITIONS, (type, segment) -> {
			if (segment != targetSegment) {
				return secureSegment(type, segment);
			}
			return segment;
		});
		return StructuredTypeRefImpl.typeRef(segments);
	}

	@Override
	public CqnPredicate where(Predicate where) {
		// Authorization condition for the statement target is added to the where() condition
		CqnPredicate authorizationCondition = authorizationService.calcWhereCondition(target.getQualifiedName(), event);
		if (authorizationCondition != null) {
			return CQL.and(where, authorizedOrInactive(target, authorizationCondition));
		}
		return where;
	}

	@Override
	public CqnPredicate exists(Select<?> subQuery) {
		return CQL.exists(CQL.copy(subQuery, newModifier(targetEntity(subQuery, model))));
	}

	@Override
	public CqnPredicate match(CqnMatchPredicate match) {
		List<Segment> segments = copyRef(match.ref());
		return CQL.match(CQL.to(segments).as(match.ref().alias().orElse(null)).asRef(),
			match.predicate().orElse(null),
			match.quantifier());
	}

	@Override
	public CqnSelectListItem expand(CqnExpand expand) {
		rejectStarAndEmptyRefs(expand.ref());

		List<Segment> segments = copyRef(expand.ref());
		CdsStructuredType expandTarget = CdsModelUtils.target(target, expand.ref().segments());
		CqnExpand result = CQL.to(segments).as(expand.ref().alias().orElse(null))
			.expand(ExpressionVisitor.copy(expand.items(), newModifier(expandTarget)))
			.limit(expand.top(), expand.skip())
			.inlineCount(expand.hasInlineCount())
			.orderBy(expand.orderBy()).as(expand.displayName());
		if (((ExpandBuilder<?>) expand).lazy()) {
			((ExpandBuilder<?>) result).lazy(true);
		}
		return result;
	}

	@Override
	public CqnSelectListItem inline(CqnInline inline) {
		rejectStarAndEmptyRefs(inline.ref());

		List<Segment> segments = copyRef(inline.ref());
		CdsStructuredType inlineTarget = CdsModelUtils.target(target, inline.ref().segments());
		return CQL.to(segments).as(inline.ref().alias().orElse(null))
			.inline(ExpressionVisitor.copy(inline.items(), newModifier(inlineTarget)));
	}

	@Override
	public CqnValue ref(CqnElementRef ref) {
		// ref with association as a single segment pulls FK only.
		if (ref.size() < 2 || CdsConstants.$USER.equals(ref.firstSegment())) {
			return ref;
		}

		// Secure ref in the select list -> path access to something
		// while avoiding eager copies of the segments. In case of element refs
		// it could be that the elements, structures are far more common than
		// the associations
		List<Segment> segments = new ArrayList<>(ref.segments());
		// Assume that $outer is always resolvable from the target of the parent type
		// E.g. `select from Order.item where exists $outer.active = true` parent target is item
		if (CqnExistsSubquery.OUTER.equals(ref.firstSegment())) {
			secureRelativeRef(parentTarget, segments);
		} else {
			secureRelativeRef(target, segments);
		}
		return CQL.get(segments).as(ref.alias().orElse(null));
	}

	private List<Segment> copyRef(CqnReference ref) {
		rejectStarAndEmptyRefs(ref);
		List<Segment> segments = new ArrayList<>(ref.segments());
		secureRelativeRef(target, segments);
		return segments;
	}

	private void secureRelativeRef(CdsStructuredType from, List<Segment> segments) {
		traverseAsRelative(from, segments, NO_COMPOSITIONS, this::secureSegment);
	}

	private Segment secureSegment(CdsStructuredType target, Segment segment) {
		CqnPredicate authorizationCondition =
			authorizedOrInactive(target, authorizationService.calcWhereCondition(target.getQualifiedName(), event));
		if (authorizationCondition != null) {
			return CQL.refSegment(segment.id(), CQL.and(segment.filter().orElse(null), authorizationCondition));
		}
		return segment;
	}

	private CqnPredicate authorizedOrInactive(CdsStructuredType target, CqnPredicate authorizationCondition) {
		if (DraftUtils.isDraftEnabled(target)) {
			return CQL.or(authorizationCondition, IS_ACTIVE_EQ_FALSE);
		}
		return authorizationCondition;
	}

	private ReadStatementAuthorizationModifier newModifier(CdsStructuredType newTarget) {
		return new ReadStatementAuthorizationModifier(authorizationService, model, event, target, newTarget);
	}

	private static void rejectStarAndEmptyRefs(CqnReference ref) {
		// Defensive, star and empty refs are rejected earlier with a proper error message
		if (ref.segments().isEmpty() || ref.firstSegment().equals("*")) {
			throw new IllegalStateException("The star expand or empty ref are not supposed to reach authorization check");
		}
	}
}
