/*********************************************************************
 * (C) 2024 SAP SE or an SAP affiliate company. All rights reserved. *
 *********************************************************************/
package com.sap.cds.services.impl.authorization;

import java.util.List;
import java.util.ListIterator;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.BiFunction;
import java.util.function.Predicate;

import com.sap.cds.ql.cqn.CqnExistsSubquery;
import com.sap.cds.ql.cqn.CqnReference.Segment;
import com.sap.cds.reflect.CdsAssociationType;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsStructuredType;

class ReferenceAssociationTraverser {

	static final Predicate<CdsElement> NO_COMPOSITIONS = e -> !e.getType().as(CdsAssociationType.class).isComposition();

	private ReferenceAssociationTraverser() {
		// hidden
	}

	/**
	 * Traverses the @{@link com.sap.cds.ql.cqn.CqnReference} represented as collection of segments and lets you
	 * replace the segments with copies.
	 * This variant treats the reference as _relative_ one - where the first segment is assumed to be a first
	 * item of the path e.g. to an element.
	 *
	 * @param root Structured type
	 * @param segments Sequence of segments that is assumed to be mutable
	 * @param takeIf boolean function to define if the association is considered
	 * @param action Some action
	 * @param <T> Assumed to be a Segment or RefSegment (immutable or mutable)
	 */
	static <T extends Segment> void traverseAsRelative(CdsStructuredType root, List<T> segments, Predicate<CdsElement> takeIf, BiFunction<CdsStructuredType, T, T> action) {
		// Prepare for UCSN -> not every segment leads to an association
		StringBuilder path = new StringBuilder();
		ListIterator<T> iterator = segments.listIterator();
		while (iterator.hasNext()) {
			root = traverseSegment(root, iterator, path, takeIf, (type, s) -> {
				T copy = action.apply(type, s);
				iterator.set(copy);
			});
		}
	}

	/**
	 * Traverses the @{@link com.sap.cds.ql.cqn.CqnReference} represented as collection of segments and lets you
	 * observe the segment
	 * This variant treats the reference as _relative_ one - where the first segment is assumed to be a first
	 * item of the path e.g. to an element.
	 * As the last argument you may supply lambda to do something useful with the segment or its target
	 *
	 * @param root Structured type
	 * @param segments Sequence of segments
	 * @param takeIf boolean function to define if the association is considered
	 * @param action Some action
	 * @param <T> Assumed to be a Segment or RefSegment (immutable or mutable)
	 */
	static <T extends Segment> void traverseAsRelative(CdsStructuredType root, List<T> segments, Predicate<CdsElement> takeIf, BiConsumer<CdsStructuredType, T> action) {
		// Prepare for UCSN -> not every segment leads to an association
		StringBuilder path = new StringBuilder();
		ListIterator<T> iterator = segments.listIterator();
		while (iterator.hasNext()) {
			root = traverseSegment(root, iterator, path, takeIf, action);
		}
	}

	/**
	 * Traverses the @{@link com.sap.cds.ql.cqn.CqnReference} represented as collection of segments and lets you
	 * replace the segment with a new one.
	 * This variant treats the reference as _absolute_ one - where the first segment is assumed to be a root type
	 * and is visited first.
	 * As the last argument you may supply lambda to do something useful with the segment or its target
	 *
	 * @param root Structured type
	 * @param segments Sequence of segments that is assumed to be mutable
	 * @param takeIf boolean function to define if the association is considered
	 * @param action Some action
	 * @param <T> Assumed to be a Segment or RefSegment (immutable or mutable)
	 */
	static <T extends Segment> void traverseAsAbsolute(CdsStructuredType root, List<T> segments, Predicate<CdsElement> takeIf, BiFunction<CdsStructuredType, T, T> action) {
		ListIterator<T> iterator = segments.listIterator();
		if (iterator.hasNext()) {
			iterator.set(action.apply(root, iterator.next()));

			StringBuilder path = new StringBuilder();
			while (iterator.hasNext()) {
				root = traverseSegment(root, iterator, path, takeIf, (type, s) -> {
					T copy = action.apply(type, s);
					iterator.set(copy);
				});
			}
		}
	}

	/**
	 * Traverses the @{@link com.sap.cds.ql.cqn.CqnReference} represented as collection of segments and lets you
	 * observe the segment.
	 * This variant treats the reference as _absolute_ one - where the first segment is assumed to be a root type
	 * and is visited first.
	 * As the last argument you may supply lambda to do something useful with the segment or its target
	 *
	 * @param root Structured type
	 * @param segments Sequence of segments
	 * @param takeIf boolean function to define if the association is considered
	 * @param action Some action
	 * @param <T> Assumed to be a Segment or RefSegment (immutable or mutable)
	 */
	static <T extends Segment> void traverseAsAbsolute(CdsStructuredType root, List<T> segments, Predicate<CdsElement> takeIf, BiConsumer<CdsStructuredType, T> action) {
		ListIterator<T> iterator = segments.listIterator();
		if (iterator.hasNext()) {
			action.accept(root, iterator.next());

			StringBuilder path = new StringBuilder();
			while (iterator.hasNext()) {
				root = traverseSegment(root, iterator, path, takeIf, action);
			}
		}
	}

	private static <T extends Segment> CdsStructuredType traverseSegment(CdsStructuredType root, ListIterator<T> segmentIter, StringBuilder path, Predicate<CdsElement> takeIf, BiConsumer<CdsStructuredType, T> action) {
		T segment = segmentIter.next();
		if (CqnExistsSubquery.OUTER.equals(segment.id())) {
			return root;
		}
		if (!path.isEmpty()) {
			path.append(".").append(segment.id());
		} else {
			path.append(segment.id());
		}

		// REVISIT: On UCSN we can throw if element is absent and skip if it is a structure
		Optional<CdsElement> element = root.findElement(path.toString());
		if (element.filter(e -> e.getType().isAssociation() && takeIf.test(e)).isPresent()) {
			// It will be our next target and the path will be recreated as relative from new target
			root = element.get().getType().as(CdsAssociationType.class).getTarget();
			path.delete(0, path.length());

			action.accept(root, segment);
		}
		return root; // Keep going
	}
}
