/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.changeset;

import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import java.util.Objects;
import java.util.Stack;
import java.util.concurrent.atomic.AtomicInteger;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.services.ServiceException;
import com.sap.cds.services.changeset.ChangeSetContextSPI;
import com.sap.cds.services.changeset.ChangeSetListener;
import com.sap.cds.services.changeset.ChangeSetMember;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;

public class ChangeSetContextImpl implements ChangeSetContextSPI {

	private static final Logger logger = LoggerFactory.getLogger(ChangeSetContextImpl.class);
	private static final AtomicInteger idProvider = new AtomicInteger();

	private boolean isTransactional;
	private boolean markedForCancel;
	private boolean isClosed;

	private final int id;
	private final boolean shadow;

	// don't need thread safety as we create context objects only thread local
	private List<ChangeSetListener> listeners = new ArrayList<>(); // maintain insertion order & use list to allow modification during iteration

	private LinkedList<ChangeSetMember> members = new LinkedList<>(); // maintain reverse insertion order

	private static final ThreadLocal<Stack<ChangeSetContextImpl>> changeSetContexts = ThreadLocal.withInitial(() -> new Stack<>());

	private ChangeSetContextImpl(boolean shadow, boolean isTransactional) {
		this.id = idProvider.incrementAndGet();
		this.shadow = shadow;
		this.isTransactional = isTransactional;
	}

	public static ChangeSetContextImpl open(boolean isTransactional) {
		return initChangeSetContext(false, isTransactional);
	}

	public static ChangeSetContextImpl attach() {
		return initChangeSetContext(true, true);
	}

	private static ChangeSetContextImpl initChangeSetContext(boolean shadow, boolean isTransactional) {
		ChangeSetContextImpl changeSetContext = new ChangeSetContextImpl(shadow, isTransactional);
		changeSetContexts.get().push(changeSetContext);

		logger.debug("Opened {}ChangeSet {}", shadow ? "shadow " : "", changeSetContext.getId());

		return changeSetContext;
	}

	public static ChangeSetContextSPI getCurrent() {
		Stack<ChangeSetContextImpl> stack = changeSetContexts.get();
		return stack.isEmpty() ? null : stack.peek();
	}

	@Override
	public int getId() {
		return id;
	}

	@Override
	public boolean isMarkedTransactional() {
		return isTransactional;
	}

	@Override
	public void markTransactional() {
		this.isTransactional = true;
	}

	@Override
	public void register(ChangeSetListener listener) {
		if (!listeners.contains(listener)) {
			listeners.add(listener);
		}
	}

	@Override
	public void register(ChangeSetMember member) {
		if(shadow && !members.isEmpty()) {
			// We can't support multiple ChangeSetMembers for shadow contexts.
			// In Spring this leads to issues, as the second transaction suspends the transaction synchronization
			// which is required to commit/rollback the overall ChangeSet correctly.
			// In Spring when using nested @Transactional annotations REQUIRES_NEW must be used to force a dedicated ChangeSet for each
			// interactions with PersistenceServices of other transaction manager is not possible in that scope.
			// Such restrictions do not occur when inside a normal ChangeSetContext scope, as we have full control of commit/rollback then.
			throw new ErrorStatusException(CdsErrorStatuses.MULTIPLE_CHANGE_SET_MEMBERS, id);
		}
		if (members.stream().anyMatch(m -> Objects.equals(m.getName(), member.getName())) || members.contains(member)) {
			throw new ErrorStatusException(CdsErrorStatuses.DUPLICATE_CHANGE_SET_MEMBERS, member.getName(), id);
		}
		members.addFirst(member);
	}

	@Override
	public boolean hasChangeSetMember(String name) {
		return members.stream().anyMatch((member) -> member.getName().equals(name));
	}

	public void triggerBeforeClose() {
		try {
			// allow modification during iteration
			// -> listeners.size() might change during iteration
			for(int i=0; i<listeners.size(); i++) {
				listeners.get(i).beforeClose();
			}
		} catch (Exception e) { // NOSONAR
			// treat this situation as reject
			// e could be ServiceException with HTTP code set or any other exception
			// TODO: what about freshly written errors?
			markForCancel();
			logger.info("Exception in listener marked the ChangeSet {} as cancelled: {}", getId(), e.getMessage());
			throw e;
		}
	}

	@Override
	public void close() {
		if (isClosed) {
			return;
		}
		isClosed = true;

		ServiceException rejectException = null;
		ServiceException closeException = null;
		try {

			// shadow contexts are managed by an external transaction manager
			// it has to take care of calling the beforeClose() listeners
			// and manage commit / rollback
			if(!shadow) {
				try {
					triggerBeforeClose();
				} catch (ServiceException e) {
					rejectException = e;
				} catch (Exception e) {
					rejectException = new ServiceException(e);
				}

				// don't allow registering new members
				ChangeSetMember[] membersCopy = members.toArray(new ChangeSetMember[0]);
				// check rollback states of members
				for (ChangeSetMember member : membersCopy) {
					if (member.isMarkedForCancel()) {
						markForCancel();
						logger.info("Rollback only status in member {} marked the ChangeSet {} as cancelled",
								member.getName(), getId());
					}
				}

				final boolean markedForCancel = isMarkedForCancel(); // save the cancel state
				logger.debug("{} ChangeSet {}", markedForCancel ? "Cancelling" : "Completing", getId());

				for (ChangeSetMember member : membersCopy) {
					try {
						if (markedForCancel) {
							member.cancel();
						} else {
							member.complete();
						}
					} catch (Exception e) { // NOSONAR
						logger.error("Unexpected exception during {} of member {} in ChangeSet {}: {}",
								markedForCancel ? "cancelation" : "completion", member.getName(), getId(), e.getMessage(), e);
						if (closeException == null) {
							// ServiceExceptions are treated as internal error during cancel/complete
							if(markedForCancel) {
								closeException = new ErrorStatusException(CdsErrorStatuses.CHANGESET_CANCELATION_FAILED, member.getName(), getId(), e);
							} else {
								closeException = new ErrorStatusException(CdsErrorStatuses.CHANGESET_COMPLETION_FAILED, member.getName(), getId(), e);
							}
						}
					}
				}

			}
		} finally {
			changeSetContexts.get().pop();
			logger.debug("Closed {}ChangeSet {}", shadow ? "shadow " : "", getId());
		}

		// no new listeners allowed be registered during afterClose()
		ChangeSetListener[] listenersCopy = listeners.toArray(new ChangeSetListener[0]);
		// call after close, after removing the current changeSet from the stack
		for (ChangeSetListener listener : listenersCopy) {
			try {
				listener.afterClose(!markedForCancel && closeException == null);
			} catch (Exception e) { // NOSONAR
				// we should not influence the response here anymore
				logger.error("Unexpected exception during afterClose of ChangeSet {}: {}", getId(), e.getMessage(), e);
			}
		}

		if (closeException != null) {
			throw closeException;
		}

		if (rejectException != null) {
			throw rejectException;
		}
	}

	boolean isClosed() {
		return isClosed;
	}

	@Override
	public void markForCancel() {
		markedForCancel = true;
	}

	@Override
	public boolean isMarkedForCancel() {
		return markedForCancel;
	}

}
