/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.draft;

import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

import com.sap.cds.impl.builder.model.ExpandBuilder;
import com.sap.cds.impl.parser.builder.ExpressionBuilder;
import com.sap.cds.impl.parser.token.CqnBoolLiteral;
import com.sap.cds.ql.CQL;
import com.sap.cds.ql.Expand;
import com.sap.cds.ql.Predicate;
import com.sap.cds.ql.RefBuilder;
import com.sap.cds.ql.RefBuilder.RefSegment;
import com.sap.cds.ql.StructuredTypeRef;
import com.sap.cds.ql.Value;
import com.sap.cds.ql.cqn.CqnElementRef;
import com.sap.cds.ql.cqn.CqnExpand;
import com.sap.cds.ql.cqn.CqnInline;
import com.sap.cds.ql.cqn.CqnReference.Segment;
import com.sap.cds.ql.cqn.CqnSelectListItem;
import com.sap.cds.ql.cqn.CqnSortSpecification;
import com.sap.cds.ql.cqn.CqnStructuredTypeRef;
import com.sap.cds.ql.cqn.CqnValue;
import com.sap.cds.ql.cqn.Modifier;
import com.sap.cds.ql.impl.ExpressionVisitor;
import com.sap.cds.reflect.CdsBaseType;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsModel;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.draft.DraftAdministrativeData;
import com.sap.cds.services.draft.Drafts;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.DraftUtils;
import com.sap.cds.services.utils.ErrorStatusException;

public class DraftModifier implements Modifier {

	public static final String DRAFT_SUFFIX = "_drafts";
	private static final String SIBLING  = "SiblingEntity";

	private final CdsEntity root;
	private final Boolean isActiveExecution;
	private final boolean addSecurityConstraints;
	private final boolean stripIsActiveEntity;
	private final EventContext context;

	private CdsEntity target;

	public DraftModifier(CdsEntity root, Boolean isActiveExecution, boolean addSecurityConstraints, boolean stripIsActiveEntity, EventContext context) {
		this.root = root;
		this.isActiveExecution = isActiveExecution;
		this.addSecurityConstraints = addSecurityConstraints;
		this.stripIsActiveEntity = stripIsActiveEntity;
		this.context = context;
	}

	private DraftModifier(DraftModifier copy, CdsEntity target) {
		this(copy.root, copy.isActiveExecution, copy.addSecurityConstraints, copy.stripIsActiveEntity, copy.context);
		this.target = target;
	}

	@Override
	public CqnStructuredTypeRef ref(CqnStructuredTypeRef original) {
		CdsEntity entity = root;
		RefBuilder<StructuredTypeRef> ref = CQL.copy(original);
		RefSegment rootSegment = ref.rootSegment();
		boolean startsWithEntity = entity.getQualifiedName().equals(rootSegment.id());
		// adapt root segment
		if (isActiveExecution != null && startsWithEntity && DraftUtils.isDraftEnabled(entity)) {
			String id = rootSegment.id();
			// manipulate from inactive to active
			if (isActiveExecution && id.endsWith(DRAFT_SUFFIX)) {
				rootSegment.id(id.substring(0, id.length() - DRAFT_SUFFIX.length()));
				entity = entity.getTargetOf(SIBLING);
			// manipulate from active to inactive
			} else if (!isActiveExecution && !id.endsWith(DRAFT_SUFFIX)) {
				rootSegment.id(id + DRAFT_SUFFIX);
				entity = entity.getTargetOf(SIBLING);
				if (addSecurityConstraints) {
					Predicate securityConstraints = getSecurityConstraints(context);
					rootSegment.filter(rootSegment.filter().map(f -> CQL.and(f, securityConstraints)).orElse(securityConstraints));
				}
			}
		}

		// adapt filters
		for (RefSegment segment: ref.segments()) {
			if (!startsWithEntity) {
				if (entity.findAssociation(segment.id()).isPresent()) {
					entity = entity.getTargetOf(segment.id());
				} else if (DraftUtils.isDraftEnabled(entity) && segment.id().equals("*")) {
					throw new ErrorStatusException(CdsErrorStatuses.UNSUPPORTED_EXPAND_ALL_IN_DRAFT);
				} else {
					break;
				}
			}
			startsWithEntity = false;
			if (segment.filter().isPresent()) {
				segment.filter(CQL.copy(segment.filter().get(), new DraftModifier(this, entity)));
			}
		}

		this.target = entity;
		return ref.build();
	}

	@Override
	public List<CqnSelectListItem> items(List<CqnSelectListItem> items) {
		if (DraftAdministrativeData.ENTITY_NAME.equals(target.getName())) {
			if (items.isEmpty()) {
				items.add(CQL.star());
			}
			if (items.stream().anyMatch(i -> i.isStar())) {
				items.add(buildInProcessByUserExpression(CQL.get(DraftAdministrativeData.IN_PROCESS_BY_USER)).as(DraftAdministrativeData.IN_PROCESS_BY_USER));
				items.add(buildDraftIsCreatedByMeExpression(CQL.get(DraftAdministrativeData.DRAFT_IS_CREATED_BY_ME)).as(DraftAdministrativeData.DRAFT_IS_CREATED_BY_ME));
				items.add(buildDraftIsProcessedByMeExpression(CQL.get(DraftAdministrativeData.DRAFT_IS_PROCESSED_BY_ME)).as(DraftAdministrativeData.DRAFT_IS_PROCESSED_BY_ME));
			}
		}
		return items;
	}

	@Override
	public CqnValue ref(CqnElementRef ref) {
		if (stripIsActiveEntity && isIsActiveEntityOnActiveTarget(ref, target)) {
			return CQL.constant(true);
		} else if (DraftAdministrativeData.IN_PROCESS_BY_USER.equals(ref.lastSegment())) {
			return buildInProcessByUserExpression(ref);
		} else if (DraftAdministrativeData.DRAFT_IS_CREATED_BY_ME.equals(ref.lastSegment())) {
			return buildDraftIsCreatedByMeExpression(ref);
		} else if (DraftAdministrativeData.DRAFT_IS_PROCESSED_BY_ME.equals(ref.lastSegment())) {
			return buildDraftIsProcessedByMeExpression(ref);
		}
		return ref;
	}

	private Value<?> buildInProcessByUserExpression(CqnElementRef ref) {
		Instant timeoutThreshold = getCancellationThreshold(context);
		CqnElementRef refLastChangeDateTime = changeLastSegment(ref, DraftAdministrativeData.LAST_CHANGE_DATE_TIME);
		CqnElementRef refDraftUuid = changeLastSegment(ref, DraftAdministrativeData.DRAFT_UUID);
		return ExpressionBuilder.create()
				.plain("CASE WHEN").ref(refDraftUuid).plain("IS NULL THEN null WHEN")
				.ref(refLastChangeDateTime).plain(">").add(CQL.constant(timeoutThreshold)).plain("THEN").ref(ref)
				.plain("ELSE '' END")
				.value().type(CdsBaseType.STRING.cdsName());
	}

	private Value<?> buildDraftIsCreatedByMeExpression(CqnElementRef ref) {
		String user = context.getUserInfo().getName();
		CqnElementRef refCreatedByUser = changeLastSegment(ref, DraftAdministrativeData.CREATED_BY_USER);
		return ExpressionBuilder.create()
				.plain("CASE WHEN").ref(refCreatedByUser).plain("=").add(CQL.val(user)).plain("THEN true")
				.plain("WHEN").ref(refCreatedByUser).plain("IS NULL").plain("THEN null ELSE false END")
				.value().type(CdsBaseType.BOOLEAN.cdsName());
	}

	private Value<?> buildDraftIsProcessedByMeExpression(CqnElementRef ref) {
		String user = context.getUserInfo().getName();
		Instant timeoutThreshold = getCancellationThreshold(context);
		CqnElementRef refCreatedByUser = changeLastSegment(ref, DraftAdministrativeData.CREATED_BY_USER);
		CqnElementRef refLastChangeDateTime = changeLastSegment(ref, DraftAdministrativeData.LAST_CHANGE_DATE_TIME);
		return ExpressionBuilder.create()
				.plain("CASE WHEN").ref(refCreatedByUser).plain("=").add(CQL.val(user))
				.plain("AND").ref(refLastChangeDateTime).plain(">").add(CQL.constant(timeoutThreshold))
				.plain("THEN true").plain("WHEN").ref(refCreatedByUser).plain("IS NULL").plain("THEN null ELSE false END")
				.value().type(CdsBaseType.BOOLEAN.cdsName());
	}

	@Override
	public CqnSelectListItem selectListValue(Value<?> value, String alias) {
		// TODO find a better solution to replace ElementRefs with expressions
		// inside of a single modifier inside where and SLVs
		// without loosing the ability to specify the alias for SLVs
		if (value.isExpression() && alias == null) {
			String type = value.type().orElse(null);
			if (CdsBaseType.STRING.cdsName().equals(type)) {
				alias = DraftAdministrativeData.IN_PROCESS_BY_USER;
			} else if (CdsBaseType.BOOLEAN.cdsName().equals(type)) {
				// TODO find a better solution
				if (value.tokens().anyMatch(t -> t.toString().equals("\"AND\""))) {
					alias = DraftAdministrativeData.DRAFT_IS_PROCESSED_BY_ME;
				} else {
					alias = DraftAdministrativeData.DRAFT_IS_CREATED_BY_ME;
				}
			}
		}
		return value.as(alias);
	}

	@Override
	public CqnSelectListItem expand(CqnExpand expand) {
		Modifier modifier = new DraftModifier(target, null, addSecurityConstraints, stripIsActiveEntity, context);
		StructuredTypeRef ref = ExpressionVisitor.copy(expand.ref(), modifier);
		List<CqnSelectListItem> items = modifier.items(expand.items().stream().map(i -> ExpressionVisitor.copy(i, modifier)).collect(Collectors.toList()));
		List<CqnSortSpecification> orderBy = modifier.orderBy(expand.orderBy().stream().map(o -> ExpressionVisitor.copy(o, modifier)).collect(Collectors.toList()));

		Expand<?> copy = CQL.to(ref.segments()).as(ref.alias().orElse(null)).expand(items).orderBy(orderBy).limit(expand.top(), expand.skip()).inlineCount(expand.hasInlineCount());
		if (((ExpandBuilder<?>) expand).lazy()) {
			((ExpandBuilder<?>) copy).lazy(true);
		}
		return copy;
	}

	@Override
	public CqnSelectListItem inline(CqnInline inline) {
		Modifier modifier = new DraftModifier(target, null, addSecurityConstraints, stripIsActiveEntity, context);
		StructuredTypeRef refCopy = ExpressionVisitor.copy(inline.ref(), modifier);
		List<CqnSelectListItem> itemsCopy = modifier.items(inline.items().stream().map(i -> ExpressionVisitor.copy(i, modifier)).collect(Collectors.toList()));
		return CQL.to(refCopy.segments()).inline(itemsCopy);
	}

	private boolean isIsActiveEntityOnActiveTarget(CqnElementRef ref, CdsEntity entity) {
		if (Drafts.IS_ACTIVE_ENTITY.equals(ref.lastSegment())) {
			for (Segment segment : ref.segments()) {
				String id = segment.id();
				if (entity.findAssociation(id).isPresent()) {
					entity = entity.getTargetOf(id);
				}
			}
			return !entity.getQualifiedName().endsWith(DRAFT_SUFFIX);
		}
		return false;
	}

	private static CqnElementRef changeLastSegment(CqnElementRef ref, String id) {
		List<Segment> segments = new ArrayList<>(ref.segments());
		segments.set(segments.size() - 1, CQL.refSegment(id));
		return CQL.get(segments);
	}

	static Instant getCancellationThreshold(EventContext context) {
		// TODO should we be able to get the NOW time stamp from the context?
		return Instant.now().minus(context.getCdsRuntime().getEnvironment().getCdsProperties().getDrafts().getCancellationTimeout()).truncatedTo(ChronoUnit.MILLIS);
	}

	static Predicate getSecurityConstraints(EventContext context) {
		if (!context.getCdsRuntime().getEnvironment().getCdsProperties().getSecurity().getAuthorization().getDraftProtection().isEnabled()
				|| context.getUserInfo().isPrivileged()) {
			return CqnBoolLiteral.TRUE;
		}
		return CQL.to(Drafts.DRAFT_ADMINISTRATIVE_DATA).anyMatch(d -> CQL.or(
			d.get(DraftAdministrativeData.CREATED_BY_USER).isNull(),
			d.get(DraftAdministrativeData.CREATED_BY_USER).eq(context.getUserInfo().getName())
		));
	}

	static CdsEntity getDraftsEntity(CdsEntity entity, CdsModel model) {
		if (entity.getName().endsWith(DraftModifier.DRAFT_SUFFIX)) {
			return entity;
		}
		return model.getEntity(entity.getQualifiedName() + DRAFT_SUFFIX);
	}

}
