/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.messaging.file;

import java.io.File;
import java.io.IOException;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.services.environment.CdsProperties.Messaging.MessagingServiceConfig;
import com.sap.cds.services.messaging.TopicMessageEventContext;
import com.sap.cds.services.messaging.service.AbstractMessagingService;
import com.sap.cds.services.messaging.service.MessagingBrokerQueueListener;
import com.sap.cds.services.runtime.CdsRuntime;

/**
 * Implementation of the {@code MessagingBrokerConnector} for the file based
 * messaging service.
 */
public class FileBasedMessagingService extends AbstractMessagingService {

	public static final String FILE_KIND = "file-based-messaging";
	private static final Logger logger = LoggerFactory.getLogger(FileBasedMessagingService.class);

	private final FileBasedMessagingBroker broker;

	public FileBasedMessagingService(MessagingServiceConfig serviceConfig, CdsRuntime runtime) throws IOException {
		super(serviceConfig, runtime);

		File file;
		if (serviceConfig.getBinding() == null) {
			file = File.createTempFile("messagexchange", ".txt");
		} else {
			file = new File(serviceConfig.getBinding());
			if (!file.exists() && !file.createNewFile()) {
				throw new IOException("The specified file '" + file.getAbsolutePath() + "' for the file based messaging service could not be created.");
			}
		}

		logger.info("Using '{}' for file-based message exchange", file.getAbsolutePath());
		this.broker = new FileBasedMessagingBroker(getName(), file, serviceConfig.getQueue().isForceListening());
	}

	@Override
	protected void removeQueue(String name) {
		// nothing to do
	}

	@Override
	protected void createQueue(String name, Map<String, Object> properties) {
		// nothing to do
	}

	@Override
	protected void createQueueSubscription(String queue, String topic) {
		broker.subscribeTopic(topic);
	}

	@Override
	protected void registerQueueListener(String queue, MessagingBrokerQueueListener listener) throws IOException {
		broker.registerListener(listener);
	}

	@Override
	protected void emitTopicMessage(String topic, TopicMessageEventContext messageEventContext) {
		broker.emitMessage(topic, messageEventContext);
	}

	@Override
	public void stop() {
		broker.stop();
	}

}
