/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.outbox.persistence;

import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Supplier;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.impl.parser.token.Jsonizer;
import com.sap.cds.ql.Insert;
import com.sap.cds.services.changeset.ChangeSetContext;
import com.sap.cds.services.changeset.ChangeSetListener;
import com.sap.cds.services.environment.CdsProperties.Outbox.OutboxServiceConfig;
import com.sap.cds.services.impl.outbox.AbstractOutboxService;
import com.sap.cds.services.impl.outbox.Messages;
import com.sap.cds.services.impl.outbox.Messages_;
import com.sap.cds.services.impl.outbox.persistence.collectors.PartitionCollector;
import com.sap.cds.services.impl.utils.CdsServiceUtils;
import com.sap.cds.services.outbox.OutboxMessageEventContext;
import com.sap.cds.services.persistence.PersistenceService;
import com.sap.cds.services.runtime.CdsRuntime;

public class PersistentOutbox extends AbstractOutboxService {

	private static final Logger LOG = LoggerFactory.getLogger(PersistentOutbox.class);

	public static final String ATTR_EVENT = "event";
	public static final String ATTR_MESSAGE = "message";

	private final TelemetryData telemetryData;
	private final PartitionCollector collector;
	private final OutboxServiceConfig config;

	// in order to keep single registration for the same change set
	private Map<ChangeSetContext, Long> changeSetContextCache = new ConcurrentHashMap<>();

	public PersistentOutbox(String name, OutboxServiceConfig config, CdsRuntime runtime, Supplier<List<String>> tenantSupplier) {
		super(name, runtime);
		this.config = config;
		this.telemetryData = config.isObservable() ? new TelemetryDataImpl(name, config.getMaxAttempts()) : TelemetryData.NOOP;
		this.collector = new PartitionCollector(runtime, this, config, tenantSupplier, telemetryData);
	}

	Collection<OutboxStatistics> getStatistics() {
		return telemetryData.getStatistics();
	}

	void init() {
		if (config.isStartCollector()) {
			start();
		}
	}

	public void start() {
		collector.start();
	}

	public void stop() {
		try {
			stop(0);
		} catch (InterruptedException e) {
			Thread.currentThread().interrupt();
		}
	}

	public void stop(long millis) throws InterruptedException {
		collector.stop(millis);
	}

	public boolean isCollectorRunning() {
		return collector.isRunning();
	}

	@Override
	protected void submit(OutboxMessageEventContext context) {
		LOG.debug("Submitting outbox message for target '{}' with event '{}'.", getName(), context.getEvent());
		persist(context);
		LOG.debug("Stored outbox message for target '{}' with event '{}'.", getName(), context.getEvent());

		ChangeSetContext changeSetContext = context.getChangeSetContext();
		changeSetContextCache.computeIfPresent(changeSetContext, (k, val) -> val + 1);

		if (!changeSetContextCache.containsKey(changeSetContext)) {
			changeSetContextCache.put(changeSetContext, 1L);

			// only store the data that is required in the changeset context listener
			String tenant = context.getUserInfo().getTenant();

			// schedule the outbox at the end of transaction
			changeSetContext.register(new ChangeSetListener() {
				@Override
				public void afterClose(boolean completed) {
					long incomingCount = changeSetContextCache.remove(changeSetContext);
					if (completed) {
						telemetryData.recordIncomingMessages(tenant, incomingCount);

						if (config.getTriggerSchedule().isEnabled()) {
							collector.unpause();
						}
					}
				}
			});
		}
	}

	private void persist(OutboxMessageEventContext context) {
		Messages message = Messages.create();
		// set the outbox target
		Map<String, Object> data = new HashMap<>();
		data.put(ATTR_MESSAGE, context.getMessage());
		data.put(ATTR_EVENT, context.getEvent());
		message.setMsg(Jsonizer.json(data));
		message.setTarget(getName());
		message.setTimestamp(context.getTimestamp());

		PersistenceService db = CdsServiceUtils.getDefaultPersistenceService(context);
		Insert insert = Insert.into(Messages_.class).entry(message);
		db.run(insert);
	}

}
