/*
 * Decompiled with CFR 0.152.
 */
package com.sap.cds.services.impl.scheduler;

import com.fasterxml.jackson.annotation.JsonGetter;
import com.fasterxml.jackson.annotation.JsonIgnore;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.sap.cds.services.environment.CdsProperties;
import com.sap.cds.services.runtime.CdsRuntime;
import java.time.Duration;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TaskScheduler
implements Runnable {
    private static final Logger LOG = LoggerFactory.getLogger(TaskScheduler.class);
    private final Map<String, TaskInfo> tasksTable = new ConcurrentHashMap<String, TaskInfo>();
    private final Object controllerMonitor = new Object();
    private final Thread controllerThread = new Thread((Runnable)this, "task-scheduler-controller");
    private final int poolSize;
    private final Duration lookupInterval;
    private volatile boolean doPause = true;
    private ThreadPoolExecutor executor;
    private SchedulerListener listener = new SchedulerListener();

    @VisibleForTesting
    TaskScheduler(int poolSize, long lookupIntervalMillis) {
        this.poolSize = poolSize;
        this.lookupInterval = Duration.ofMillis(lookupIntervalMillis);
        this.start();
    }

    public TaskScheduler(CdsRuntime runtime) {
        CdsProperties config = runtime.getEnvironment().getCdsProperties();
        this.poolSize = config.getTaskScheduler().getThreadPoolSize();
        this.lookupInterval = config.getTaskScheduler().getLookupInterval();
    }

    public void start() {
        this.executor = new ThreadPoolExecutor(this.poolSize, this.poolSize, 1L, TimeUnit.MINUTES, new LinkedBlockingQueue<Runnable>(), new ThreadFactoryBuilder().setNameFormat("task-scheduler-%d").setDaemon(true).build());
        this.controllerThread.setDaemon(true);
        this.controllerThread.start();
        LOG.info("Started Task Scheduler with pool size '{}' and lookup interval '{}'", (Object)this.poolSize, (Object)this.lookupInterval);
    }

    public void shutdown() {
        this.controllerThread.interrupt();
        this.executor.shutdownNow();
    }

    public void scheduleTask(String key, Task task, long inMillis) {
        if (this.executor.isShutdown()) {
            return;
        }
        this.listener.scheduledNewTask(this.tasksTable.compute(key, (k, taskInfo) -> {
            if (taskInfo == null) {
                LOG.debug("Scheduling new task '{}' in {} millis", (Object)key, (Object)inMillis);
                return new TaskInfo(key, System.currentTimeMillis() + inMillis, task);
            }
            if (taskInfo.setScheduleTsIfLower(System.currentTimeMillis() + inMillis)) {
                LOG.debug("Reduced wait time for existing task '{}' to {} millis", (Object)key, (Object)inMillis);
            }
            return taskInfo;
        }));
        this.wakeupScheduler();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void wakeupScheduler() {
        if (this.doPause) {
            Object object = this.controllerMonitor;
            synchronized (object) {
                if (this.doPause) {
                    this.controllerMonitor.notifyAll();
                    this.doPause = false;
                }
            }
        }
    }

    @Override
    public void run() {
        long lastMinInterval = this.lookupInterval.toMillis();
        block2: while (this.await(lastMinInterval)) {
            long currentTs = System.currentTimeMillis();
            lastMinInterval = this.lookupInterval.toMillis();
            for (TaskInfo taskInfo : this.tasksTable.values()) {
                if (taskInfo.submitted || !taskInfo.task.isReady()) continue;
                long scheduleTs = taskInfo.scheduleTs;
                if (scheduleTs <= currentTs) {
                    try {
                        this.submitTask(taskInfo);
                        continue;
                    }
                    catch (RejectedExecutionException e) {
                        continue block2;
                    }
                }
                lastMinInterval = Math.min(lastMinInterval, scheduleTs - currentTs);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean await(long pauseTime) {
        Object object = this.controllerMonitor;
        synchronized (object) {
            try {
                if (this.doPause) {
                    this.listener.sleeping(pauseTime);
                    this.controllerMonitor.wait(pauseTime);
                }
                this.doPause = true;
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                return false;
            }
        }
        return !this.executor.isShutdown();
    }

    private void submitTask(TaskInfo taskInfo) {
        LOG.debug("Submitting task '{}'", (Object)taskInfo.key);
        taskInfo.submitted = true;
        this.executor.submit(() -> {
            TaskSchedule result = null;
            try {
                LOG.debug("Starting task '{}'", (Object)taskInfo.key);
                this.listener.taskStarted(taskInfo);
                result = taskInfo.task.run();
            }
            finally {
                LOG.debug("Task '{}' finished", (Object)taskInfo.key);
                if (result == null) {
                    this.tasksTable.remove(taskInfo.key);
                } else {
                    LOG.debug("Rescheduling task '{}' in {} millis", (Object)taskInfo.key, (Object)result.delay);
                    taskInfo.setScheduleTs(System.currentTimeMillis() + result.delay, result.allowEarlier);
                    taskInfo.task = result.task;
                    taskInfo.submitted = false;
                    this.wakeupScheduler();
                }
                this.listener.taskFinished(taskInfo);
            }
        });
        this.listener.submittedNewTask(taskInfo);
    }

    public void setListener(SchedulerListener listener) {
        this.listener = listener;
    }

    public List<TaskInfo> getTasksSchedule() {
        return this.tasksTable.values().stream().sorted((o1, o2) -> (int)(o1.scheduleTs - o2.scheduleTs)).toList();
    }

    public static class SchedulerListener {
        protected void scheduledNewTask(TaskInfo info) {
        }

        public void submittedNewTask(TaskInfo taskInfo) {
        }

        public void sleeping(long pauseTime) {
        }

        protected void taskFinished(TaskInfo taskInfo) {
        }

        protected void taskStarted(TaskInfo taskInfo) {
        }
    }

    public static interface Task {
        public TaskSchedule run();

        default public boolean isReady() {
            return true;
        }
    }

    public static class TaskInfo {
        private final String key;
        private volatile long scheduleTs;
        private volatile boolean allowEarlier = true;
        private volatile boolean submitted;
        @JsonIgnore
        private volatile Task task;

        TaskInfo(String key, long scheduleTs, Task task) {
            this.key = key;
            this.task = task;
            this.scheduleTs = scheduleTs;
        }

        private synchronized void setScheduleTs(long newScheduleTs, boolean allowEarlier) {
            this.allowEarlier = allowEarlier;
            this.scheduleTs = newScheduleTs;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private boolean setScheduleTsIfLower(long newScheduleTs) {
            if (newScheduleTs < this.scheduleTs) {
                TaskInfo taskInfo = this;
                synchronized (taskInfo) {
                    if (this.allowEarlier && newScheduleTs < this.scheduleTs) {
                        this.scheduleTs = newScheduleTs;
                        return true;
                    }
                }
            }
            return false;
        }

        @JsonGetter(value="suspended")
        public boolean isSuspended() {
            return !this.task.isReady();
        }

        @JsonGetter(value="key")
        public String getKey() {
            return this.key;
        }

        @JsonGetter(value="scheduleTs")
        public long getScheduleTs() {
            return this.scheduleTs;
        }

        @JsonGetter(value="submitted")
        public boolean getSubmmited() {
            return this.submitted;
        }
    }

    public record TaskSchedule(Task task, long delay, boolean allowEarlier) {
        public TaskSchedule(Task task) {
            this(task, 0L, false);
        }
    }
}

