/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.auditlog;

import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.BiConsumer;
import java.util.stream.StreamSupport;

import com.google.common.annotations.VisibleForTesting;
import com.sap.cds.CdsData;
import com.sap.cds.CdsDiffProcessor;
import com.sap.cds.impl.DataProcessor;
import com.sap.cds.impl.diff.DiffProcessor;
import com.sap.cds.ql.cqn.Path;
import com.sap.cds.reflect.CdsAssociationType;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsStructuredType;
import com.sap.cds.services.auditlog.Action;
import com.sap.cds.services.auditlog.ChangedAttribute;
import com.sap.cds.services.auditlog.DataModification;
import com.sap.cds.services.runtime.CdsRuntime;

class PersonalDataAnalyzerModification extends PersonalDataAnalyzer {
	private List<? extends Map<String, Object>> oldData;

	@VisibleForTesting
	static final String REDACTED = "*****";

	PersonalDataAnalyzerModification(CdsStructuredType entity, PersonalDataCaches caches, CdsRuntime runtime) {
		super(entity, caches, runtime);
	}

	PersonalDataAnalyzerModification(CdsEntity entity, List<? extends Map<String, Object>> data, PersonalDataCaches caches,
		CdsRuntime runtime) {
		this(entity, caches, runtime);
		setData(data);
	}

	/**
	 * Sets the {@link List} with old or deleted data for current entity.
	 *
	 * @param oldData a {@link List} with old or deleted data
	 */
	public void setOldData(List<? extends CdsData> oldData) {
		this.oldData = oldData;
	}

	/**
	 * @return a {@link List} with modifications to personal data fields.
	 */
	public List<DataModification> getDataModifications() {
		if (Boolean.TRUE.equals(personalData.getLogInsert().isEnabled())
			|| Boolean.TRUE.equals(personalData.getLogUpdate().isEnabled())
			|| Boolean.TRUE.equals(personalData.getLogDelete().isEnabled())) {

			DataToModifications visitor = new DataToModifications(this.caches);
			DiffProcessor.create().forDeepTraversal()
				.add((path, element, type) -> {
					PersonalDataMeta meta = visitor.metadataProvider.getMeta(getTarget(element, path));
					if (meta != null) {
						if (element != null && !element.getType().isAssociation()) {
							return meta.getPersonalDataNames().contains(element.getName());
						} else {
							return meta.hasPersonalData();
						}
					}
					return false;
				}, visitor)
				.process(super.data, oldData, this.entity);
			return visitor.getResult();
		} else {
			return Collections.emptyList();
		}
	}

	/**
	 * Indicates whether the given {@link CdsEntity entity} contains personal data or not.
	 *
	 * @return <code>true</code> if given {@link CdsEntity entity} contains personal data.
	 */
	@Override
	public boolean hasPersonalData() {
		if (super.data != null) {
			return hasPersonalData(super.entity, super.data);
		}

		return getMeta(super.entity).hasPersonalData();
	}

	/**
	 * Indicates whether the given data contains personal data or not.
	 *
	 * @param entity the {@link CdsStructuredType entity} whose personal data fields will be determined
	 * @param data   the data that be checked for personal data
	 * @return <code>true</code> if personal data is found
	 */
	private boolean hasPersonalData(CdsStructuredType entity, Iterable<? extends Map<String, Object>> data) {
		AtomicBoolean hasPersonalData = new AtomicBoolean(false);

		DataProcessor.create().action(new DataProcessor.Action() {
			@Override
			public void entries(Path path, CdsElement element, CdsStructuredType type, Iterable<Map<String, Object>> dataList) {
				PersonalDataMeta meta = getMeta(type);

				if (meta != null) {
					Set<String> personal = meta.getPersonalDataNames();

					// check if we found already personal data in the provided data tree
					if (!hasPersonalData.get() && meta.hasPersonalData() && StreamSupport.stream(dataList.spliterator(), false)
						.anyMatch(dataRow -> dataRow.keySet().stream().anyMatch(personal::contains))) {
						hasPersonalData.set(true);
					}
				}
			}
		}).process(data, entity);

		return hasPersonalData.get();
	}

	private class DataToModifications implements CdsDiffProcessor.DiffVisitor {

		private final PersonalDataCaches metadataProvider;

		// Path is not 100% safe for the keys (instances are recreated). It is not a problem though
		// as we fetch values from here with a keys only for changed elements, where paths are stable.
		// Elements reuse DataModification instances -> visitor will gather changed elements per changed place in the entity
		private final Map<Path, DataModification> elementChanges = new HashMap<>();
		// Rest is visited at most once, no need to have a key at all
		private final List<DataModification> structureChanges = new LinkedList<>();

		DataToModifications(PersonalDataCaches metadataProvider) {
			this.metadataProvider = metadataProvider;
		}

		@Override
		public void changed(Path newPath, Path ignored, CdsElement element, Object newValue, Object oldValue) {
			if (Boolean.TRUE.equals(personalData.getLogUpdate().isEnabled())) {
				DataModification modification = elementChanges.computeIfAbsent(newPath, p ->
					newModification(Action.UPDATE, p.target().values(), metadataProvider.getMeta(p.target().type())));

				ChangedAttribute attribute = ChangedAttribute.create();
				attribute.setName(element.getName());

				boolean isSensitive = metadataProvider.getMeta(newPath.target().type()).getSensitiveDataNames().contains(element.getName());
				// nulls in the payload are not needed, audit log treats absence of a value
				// in the payload as marker that something was added or removed
				if (newValue != null) {
					attribute.setNewValue(isSensitive ? REDACTED : newValue.toString());
				}
				if (oldValue != null) {
					attribute.setOldValue(isSensitive ? REDACTED : oldValue.toString());
				}

				modification.getAttributes().add(attribute);
			}
		}

		@Override
		public void added(Path newPath, Path ignored, CdsElement association, Map<String, Object> newValue) {
			if (Boolean.TRUE.equals(personalData.getLogInsert().isEnabled())) {
				PersonalDataMeta meta = metadataProvider.getMeta(getTarget(association, newPath));
				DataModification modification =
					newModification(Action.CREATE, newValue, meta);
				withAttributes(modification, newValue, meta, ChangedAttribute::setNewValue);
				structureChanges.add(modification);
			}
		}

		@Override
		public void removed(Path newPath, Path ignored, CdsElement association, Map<String, Object> oldValue) {
			// As long as the Path is used as a key, it does not matter which one it is.
			// In case of the deep removal, the new path will be empty (values() and filter() will return nothing)
			if (Boolean.TRUE.equals(personalData.getLogDelete().isEnabled())) {
				PersonalDataMeta meta = metadataProvider.getMeta(getTarget(association, newPath));
				DataModification modification =
					newModification(Action.DELETE, oldValue, meta);
				withAttributes(modification, oldValue, meta, ChangedAttribute::setOldValue);
				structureChanges.add(modification);
			}
		}

		public List<DataModification> getResult() {
			LinkedList<DataModification> result = new LinkedList<>(elementChanges.values());
			result.addAll(structureChanges);
			return result;
		}

		private void withAttributes(DataModification modification,
			Map<String, Object> data, PersonalDataMeta meta, BiConsumer<ChangedAttribute, String> action) {

			modification.setAttributes(
				meta.getPersonalDataNames().stream()
					.filter(e -> data.get(e) != null)
					.map(e -> {
						ChangedAttribute attribute = ChangedAttribute.create();
						attribute.setName(e);
						if (meta.getSensitiveDataNames().contains(e)) {
							action.accept(attribute, REDACTED);
						} else {
							action.accept(attribute, data.get(e).toString());
						}
						return attribute;
					}).toList());
		}

		private DataModification newModification(Action action, Map<String, Object> data, PersonalDataMeta meta) {
			DataModification modification = DataModification.create();
			modification.setAction(action);
			modification.setDataObject(createDataObject(data, meta));
			modification.setDataSubject(createDataSubject(data, meta));
			modification.setAttributes(new LinkedList<>());
			return modification;
		}
	}

	private static CdsStructuredType getTarget(CdsElement element, Path path) {
		if (element != null && element.getType().isAssociation()) {
			return element.getType().as(CdsAssociationType.class).getTarget();
		}
		return path.target().type();
	}
}
