/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.authorization;

import java.util.ArrayList;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.ql.CQL;
import com.sap.cds.ql.cqn.CqnPredicate;
import com.sap.cds.reflect.CdsModel;
import com.sap.cds.services.authorization.ActionAccessEventContext;
import com.sap.cds.services.authorization.AuthorizationService;
import com.sap.cds.services.authorization.CalcWhereConditionEventContext;
import com.sap.cds.services.authorization.EntityAccessEventContext;
import com.sap.cds.services.authorization.FunctionAccessEventContext;
import com.sap.cds.services.authorization.GetRestrictionEventContext;
import com.sap.cds.services.authorization.ServiceAccessEventContext;
import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.HandlerOrder;
import com.sap.cds.services.handler.annotations.On;
import com.sap.cds.services.handler.annotations.ServiceName;
import com.sap.cds.services.impl.authorization.PredicateResolver.MultipleAttributeValuesNotSupportedException;
import com.sap.cds.services.impl.utils.CdsModelUtils;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.OrderConstants;
import com.sap.cds.services.utils.StringUtils;
import com.sap.cds.services.utils.TenantAwareCache;
import com.sap.cds.services.utils.model.Privilege;
import com.sap.cds.services.utils.model.Restriction;

@ServiceName(value = "*", type = AuthorizationService.class)
public class AuthorizationDefaultOnHandler implements EventHandler {

	private final static Logger logger = LoggerFactory.getLogger(AuthorizationDefaultOnHandler.class);

	private final TenantAwareCache<RestrictionLookup, CdsModel> restrictionLookupCache;
	private final TenantAwareCache<PredicateLookup, CdsModel> predicateLookupCache;

	private final boolean isEmptyAttributeValuesAreRestricted;

	@SuppressWarnings("deprecation")
	AuthorizationDefaultOnHandler(CdsRuntime runtime) {
		restrictionLookupCache = TenantAwareCache.create(RestrictionLookup::new, runtime);
		predicateLookupCache = TenantAwareCache.create(PredicateLookup::new, runtime);

		isEmptyAttributeValuesAreRestricted = runtime.getEnvironment().getCdsProperties().getSecurity()
				.getAuthorization().isEmptyAttributeValuesAreRestricted();
	}

	@On
	@HandlerOrder(OrderConstants.On.DEFAULT_ON)
	protected void defaultGetRestriction(GetRestrictionEventContext context) {
		Restriction restriction = switch (context.getKind()) {
		case SERVICE -> restrictionLookupCache.findOrCreate().retrieveServiceRestriction(context.getModel(), context.getName());
		case ENTITY -> restrictionLookupCache.findOrCreate().retrieveEntityRestriction(context.getModel(), context.getName());
		case ACTION -> restrictionLookupCache.findOrCreate().lookupActionRestriction(context.getModel(),
				context.getName(), context.getEventName());
		case FUNCTION -> restrictionLookupCache.findOrCreate().lookupFunctionRestriction(context.getModel(),
				context.getName(), context.getEventName());
		default -> throw new ErrorStatusException(CdsErrorStatuses.UNSUPPORTED_RESTRICTION, context.getName(),
				context.getKind());
		};
		context.setResult(restriction);
	}

	@On
	@HandlerOrder(OrderConstants.On.DEFAULT_ON)
	protected void defaultHasServiceAccess(ServiceAccessEventContext context) {

		final String event = context.getAccessEventName();
		final String service = context.getAccessServiceName();

		boolean result = true; // as long as there is no restriction access is granted

		Restriction restriction = RestrictionUtils.getServiceRestriction(context.getService(), service, event);
		if (restriction != null) {
			result = RestrictionUtils.passesRestriction(restriction, context.getUserInfo(), event);
		}

		context.setResult(result);
	}

	@On
	@HandlerOrder(OrderConstants.On.DEFAULT_ON)
	protected void defaultHasEntityAccess(EntityAccessEventContext context) {
		boolean result = true;

		final String event = context.getAccessEventName();
		final String authorizationEntityName = context.getAccessEntityName();

		// check static access to authorization entity in any case (target or path)
		Restriction restriction = RestrictionUtils.getEntityRestriction(context.getService(), authorizationEntityName,
				event);
		if (restriction != null && !RestrictionUtils.passesRestriction(restriction, context.getUserInfo(), event)) {
			logger.debug("No authorization to send event '{}' to entity '{}'", event, authorizationEntityName);
			result = false;
		} // no further restrictions

		context.setResult(result);
	}

	@On
	@HandlerOrder(OrderConstants.On.DEFAULT_ON)
	protected void defaultHasFunctionAccess(FunctionAccessEventContext context) {

		final String functionName = context.getFunctionName();
		final String entityName = context.getEntityName();

		boolean result = true; // as long as there is no restriction access is granted

		Restriction restriction = RestrictionUtils.getFunctionRestriction(context.getService(), entityName,
				functionName);
		if (restriction != null) {
			result = RestrictionUtils.passesRestriction(restriction, context.getUserInfo(),
					Privilege.PredefinedGrant.ALL
							.toString() /* = event: restrictions on functions have implicitly grant '*' */);
		}

		context.setResult(result);
	}

	@On
	@HandlerOrder(OrderConstants.On.DEFAULT_ON)
	protected void defaultHasActionAccess(ActionAccessEventContext context) {

		final String actionName = context.getActionName();
		final String entityName = context.getEntityName();

		boolean result = true; // as long as there is no restriction access is granted

		Restriction restriction = RestrictionUtils.getActionRestriction(context.getService(), entityName, actionName);
		if (restriction != null) {
			result = RestrictionUtils.passesRestriction(restriction, context.getUserInfo(),
					Privilege.PredefinedGrant.ALL
							.toString() /* = event: restrictions on actions have implicitly grant '*' */);
		}

		context.setResult(result);
	}

	@On
	@HandlerOrder(OrderConstants.On.DEFAULT_ON)
	protected void defaultCalcWhereCondition(CalcWhereConditionEventContext context) {

		final String event = context.getEventName();
		final String entityName = context.getEntityName(); // entityName is always not null.

		CqnPredicate result = null; // no filter as long there is no where condition hit

		List<Restriction> restrictions = new ArrayList<>(2);
		boolean isStandardCdsEvent = CdsModelUtils.isStandardCdsEvent(event);

		// Add the restriction of the entity itself for the event if existing
		Restriction modelRestriction = RestrictionUtils.getEntityRestriction(context.getService(), entityName, event);
		if (modelRestriction != null) {
			restrictions.add(modelRestriction);
		}

		// Add the restriction on bound action/function level if existing for the custom event
		if (!isStandardCdsEvent) {
			if (CdsModelUtils.getAction(context.getCdsRuntime().getCdsModel(), entityName, event).isPresent()) {
				modelRestriction = RestrictionUtils.getActionRestriction(context.getService(), entityName, event);
				if (modelRestriction != null) {
					restrictions.add(modelRestriction);
				}
			} else if (CdsModelUtils.getFunction(context.getCdsRuntime().getCdsModel(), entityName, event)
					.isPresent()) {
				modelRestriction = RestrictionUtils.getFunctionRestriction(context.getService(), entityName, event);
				if (modelRestriction != null) {
					restrictions.add(modelRestriction);
				}
			}
		}

		// Conjunction of the restrictions and
		// disjunction of the privileges within a restriction
		for (Restriction restriction : restrictions) {
			CqnPredicate restrictionResult = null;
			List<Privilege> privileges = RestrictionUtils
					.passingPrivilegesOfRestriction(restriction, context.getUserInfo(), event).toList();
			for (Privilege privilege : privileges) {
				CqnPredicate cqnPredicate = null;
				if (privilege.hasWhere()) {
					try {
						logger.debug("Resolving 'where' condition '{}'", privilege.getCxnWhereCondition());
						cqnPredicate = predicateLookupCache.findOrCreate().resolvePredicate(privilege,
								context.getUserInfo(), isEmptyAttributeValuesAreRestricted);

					} catch (MultipleAttributeValuesNotSupportedException e) {
						logger.debug(
								"No authorization to send event '{}' to entity '{}' because user {} has multiple values for attribute '{}' (filter resource '{}')",
								event, entityName, context.getUserInfo().getName(), e.getAttributeName(),
								e.getResourceName(), e);
						throw new ErrorStatusException(CdsErrorStatuses.EVENT_FORBIDDEN_UNSUPPORTED_USER_ATTRIBUTES,
								event, entityName, e.getAttributeName(), e);

					} catch (Exception e) { // NOSONAR
						throw new ErrorStatusException(CdsErrorStatuses.INVALID_WHERE_CONDITION,
								privilege.getCxnWhereCondition(), entityName, event, context.getUserInfo().getName(),
								e);
					}
				} else if (!StringUtils.isEmpty(privilege.getWhereCQL())) {
					// no _where property (Cxn) generated for where clause, e.g. compilation error silently ignored
					throw new ErrorStatusException(CdsErrorStatuses.INCONSISTENT_WHERE_CONDITION,
							privilege.getWhereCQL(), entityName);
				}

				if (cqnPredicate == null) { // null evaluates to 'true'
					restrictionResult = null;
					break;
				}
				if (restrictionResult != null) {
					restrictionResult = CQL.or(restrictionResult, cqnPredicate);
				} else {
					restrictionResult = cqnPredicate;
				}
			}
			if (result != null) {
				result = restrictionResult != null ? CQL.and(result, restrictionResult) : result;
			} else {
				result = restrictionResult;
			}
		}

		context.setResult(result);
	}

}
