/*********************************************************************
 * (C) 2024 SAP SE or an SAP affiliate company. All rights reserved. *
 *********************************************************************/
package com.sap.cds.services.impl.authorization;

import static com.sap.cds.services.impl.authorization.ReferenceAssociationTraverser.NO_COMPOSITIONS;
import static com.sap.cds.services.impl.authorization.ReferenceAssociationTraverser.consume;

import java.util.Collection;
import java.util.HashSet;
import java.util.Objects;
import java.util.Set;
import java.util.function.Consumer;

import com.sap.cds.ql.cqn.CqnElementRef;
import com.sap.cds.ql.cqn.CqnExistsSubquery;
import com.sap.cds.ql.cqn.CqnExpand;
import com.sap.cds.ql.cqn.CqnInline;
import com.sap.cds.ql.cqn.CqnMatchPredicate;
import com.sap.cds.ql.cqn.CqnReference;
import com.sap.cds.ql.cqn.CqnSelect;
import com.sap.cds.ql.cqn.CqnStatement;
import com.sap.cds.ql.cqn.CqnStructuredTypeRef;
import com.sap.cds.ql.cqn.CqnVisitor;
import com.sap.cds.ql.cqn.ResolvedSegment;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsModel;
import com.sap.cds.reflect.CdsStructuredType;
import com.sap.cds.reflect.impl.reader.model.CdsConstants;
import com.sap.cds.services.EventContext;
import com.sap.cds.util.CdsModelUtils;
import com.sap.cds.util.CqnStatementUtils;

public class StaticAuthorizationsDeepChecker implements CqnVisitor {

	private final CdsModel model;
	private final CdsStructuredType current;
	private final CdsStructuredType parent;
	private final Set<String> checkedEntities;
	private final Consumer<CdsEntity> authCheck;

	public StaticAuthorizationsDeepChecker(EventContext context, CqnStatement statement, Collection<String> alreadyChecked, Consumer<CdsEntity> authCheck) {
		this(context.getModel(),
			null, // No parent on root call
			CqnStatementUtils.targetEntity(statement, context.getModel()),
			// We might not be able to guarantee certain order of the events in all cases
			// visitor has a certain order of traversal, but the result and the order will
			// vary from statement to statement
			new HashSet<>(alreadyChecked), authCheck);
	}

	private StaticAuthorizationsDeepChecker(CdsModel model, CdsStructuredType parent, CdsStructuredType current, Set<String> visitedEntities, Consumer<CdsEntity> authCheck) {
		this.model = Objects.requireNonNull(model);
		this.current = Objects.requireNonNull(current);
		this.parent = parent;
		this.checkedEntities = Objects.requireNonNull(visitedEntities);
		this.authCheck = Objects.requireNonNull(authCheck);
	}

	@Override
	public void visit(CqnMatchPredicate match) {
		consume(current, match.ref(), NO_COMPOSITIONS, this::checkIfAuthorized);
		match.predicate().ifPresent(p -> p.accept(this));
	}

	@Override
	public void visit(CqnSelect select) {
		// accept(this) is not an option, we need to visit parts of the statement
		select.dispatch(newVisitor(CqnStatementUtils.targetEntity(select, model)));
	}

	@Override
	public void visit(CqnStructuredTypeRef ref) {
		CdsEntity root = model.getEntity(ref.firstSegment());
		consume(root, ref, NO_COMPOSITIONS, this::checkIfAuthorized);
	}

	@Override
	public void visit(CqnElementRef ref) {
		if (ref.size() < 2 || CdsConstants.$USER.equals(ref.firstSegment())) {
			return;
		}

		CdsStructuredType target = getRefRootType(ref);
		consume(target, ref, NO_COMPOSITIONS, this::checkIfAuthorized);
	}

	@Override
	public void visit(CqnInline inline) {
		consume(current, inline.ref(), NO_COMPOSITIONS, this::checkIfAuthorized);
		CqnVisitor visitor = newVisitor(CdsModelUtils.target(current, inline.ref().segments()));
		inline.items().forEach(i -> i.accept(visitor));
	}

	@Override
	public void visit(CqnExpand expand) {
		// Alternative: use CqnEntitySelector, but that is more obscure
		consume(current, expand.ref(), NO_COMPOSITIONS, this::checkIfAuthorized);
		CqnVisitor visitor = newVisitor(CdsModelUtils.target(current, expand.ref().segments()));
		expand.items().forEach(i -> i.accept(visitor));
		expand.orderBy().forEach(i -> i.accept(visitor));
	}

	@Override
	public void visit(CqnExistsSubquery exists) {
		exists.subquery().accept(newVisitor(CqnStatementUtils.targetEntity(exists.subquery(), model)));
	}

	private void checkIfAuthorized(ResolvedSegment current) {
		if (!checkedEntities.contains(current.type().getQualifiedName())) {
			authCheck.accept(current.entity());
			checkedEntities.add(current.type().getQualifiedName());
		}
	}

	private CqnVisitor newVisitor(CdsStructuredType next) {
		return new StaticAuthorizationsDeepChecker(model, current, next, checkedEntities, authCheck);
	}

	private CdsStructuredType getRefRootType(CqnReference ref) {
		if (CqnExistsSubquery.OUTER.equals(ref.firstSegment())) {
			return parent;
		} else {
			return current;
		}
	}

}
