/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.cds;

import static com.sap.cds.services.utils.model.CqnUtils.addWhere;

import com.sap.cds.ql.Select;
import com.sap.cds.ql.cqn.AnalysisResult;
import com.sap.cds.ql.cqn.CqnFilterableStatement;
import com.sap.cds.ql.cqn.CqnSelect;
import com.sap.cds.ql.cqn.CqnUpdate;
import com.sap.cds.ql.cqn.CqnDelete;
import com.sap.cds.ql.cqn.CqnPredicate;
import com.sap.cds.ql.cqn.CqnStatement;
import com.sap.cds.ql.cqn.CqnAnalyzer;
import com.sap.cds.services.impl.utils.CdsServiceUtils;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.ql.CQL;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.authorization.AuthorizationService;
import com.sap.cds.services.cds.ApplicationService;
import com.sap.cds.services.cds.CdsDeleteEventContext;
import com.sap.cds.services.cds.CdsReadEventContext;
import com.sap.cds.services.cds.CdsUpdateEventContext;
import com.sap.cds.services.cds.CqnService;
import com.sap.cds.services.draft.DraftEditEventContext;
import com.sap.cds.services.draft.DraftSaveEventContext;
import com.sap.cds.services.draft.DraftService;
import com.sap.cds.services.draft.Drafts;
import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.Before;
import com.sap.cds.services.handler.annotations.HandlerOrder;
import com.sap.cds.services.handler.annotations.ServiceName;
import com.sap.cds.services.impl.authorization.ReadStatementAuthorizationModifier;
import com.sap.cds.services.impl.utils.CdsModelUtils;
import com.sap.cds.services.utils.DraftUtils;
import com.sap.cds.services.utils.OrderConstants;
import com.sap.cds.services.utils.model.CqnUtils;
import com.sap.cds.util.CqnStatementUtils;

import java.util.Map;
import java.util.Optional;

@ServiceName(value = "*", type = ApplicationService.class)
public class InstanceBasedAuthorizationHandler implements EventHandler {

	private static final Logger logger = LoggerFactory.getLogger(InstanceBasedAuthorizationHandler.class);

	private final boolean authorizeDeeply;
	private final boolean regardCustomEvents;
	private final boolean rejectUnauthorized;

	public InstanceBasedAuthorizationHandler(CdsRuntime runtime) {
		var authorization = runtime.getEnvironment().getCdsProperties().getSecurity().getAuthorization();
		authorizeDeeply = authorization.getDeep().isEnabled();
		regardCustomEvents = authorization.getInstanceBased().getCustomEvents().isEnabled();
		rejectUnauthorized = authorization.getInstanceBased().getRejectSelectedUnauthorizedEntity().isEnabled();
	}

	// Restriction annotations from underlying projected entities are added to the service entity by the compiler provided that no restrictions are defined already.
	// So the authorization has to be tested only on service level.
	@Before(event = "*")
	@HandlerOrder(OrderConstants.Before.CHECK_INSTANCE_BASED_AUTHORIZATION)
	private void checkAuthorization(EventContext context) {
		// privileged users just skip authorization checks
		if ( !context.getUserInfo().isPrivileged() ) {
			extendStatements(context);
		}
	}

	private void extendStatements(EventContext context) {
		AuthorizationService authService = context.getServiceCatalog().getService(AuthorizationService.class, AuthorizationService.DEFAULT_NAME);

		final String event = context.getEvent();
		final String serviceName = ((ApplicationService) context.getService()).getDefinition().getQualifiedName();

		CdsEntity targetEntity = context.getTarget();
		String entityName = null;
		if (targetEntity != null) {
			entityName = targetEntity.getQualifiedName();
		}

		if (event.equals(CqnService.EVENT_READ)) {
			CdsReadEventContext cdsContext = context.as(CdsReadEventContext.class);
			if (authorizeDeeply) {
				CqnSelect secured = CQL.copy(cdsContext.getCqn(), new ReadStatementAuthorizationModifier(authService, cdsContext));
				if (logger.isDebugEnabled()) {
					logger.debug("Statement is extended with security conditions for target '{}' of service '{}': '{}'",
						entityName, serviceName, CqnStatementUtils.anonymizeStatement(secured));
				}
				cdsContext.setCqn(secured);
			} else {
				CqnSelect select = cdsContext.getCqn();
				CqnPredicate wherePred = authService.calcWhereCondition(entityName, CqnService.EVENT_READ);
				if (wherePred != null) {
					if (targetEntity != null && DraftUtils.isDraftEnabled(context.getTarget())) {
						wherePred = CQL.or(wherePred, CQL.get(Drafts.IS_ACTIVE_ENTITY).eq(false));
					}
					select = CqnUtils.addWhere(select, wherePred);
				}
				cdsContext.setCqn(select);
			}
		}

		// process instance-based authorization for some standard CDS events by appending the CQN where statements with additional restrictions
		if (event.equals(CqnService.EVENT_DELETE) ||
			event.equals(CqnService.EVENT_UPDATE) ||
			event.equals(DraftService.EVENT_DRAFT_EDIT) ||
			event.equals(DraftService.EVENT_DRAFT_SAVE)) {

			// finally handle where predicates for some standard cds events
			CqnPredicate wherePred = authService.calcWhereCondition(entityName, event);
			if (wherePred != null) {
				if (rejectUnauthorized) {
					// check before WHERE condition is modified
					checkEntityAuthorization(context, wherePred);
				}

				if (context.getEvent().equals(CqnService.EVENT_DELETE)) {
					CdsDeleteEventContext deleteContext = context.as(CdsDeleteEventContext.class);
					if (targetEntity != null && DraftUtils.isDraftEnabled(targetEntity)) {
						// omit wherePred for draft entities. wherePred still applies for active entities.
						wherePred = CQL.or(wherePred, CQL.get(Drafts.IS_ACTIVE_ENTITY).eq(false));
					}
					CqnDelete cqn = addWhere(deleteContext.getCqn(), wherePred);
					deleteContext.setCqn(cqn);

				} else if (context.getEvent().equals(CqnService.EVENT_UPDATE)) {
					CdsUpdateEventContext updateContext = context.as(CdsUpdateEventContext.class);
					if (targetEntity != null && DraftUtils.isDraftEnabled(targetEntity)) {
						// omit wherePred for draft entities. wherePred still applies for active entities.
						wherePred = CQL.or(wherePred, CQL.get(Drafts.IS_ACTIVE_ENTITY).eq(false));
					}
					CqnUpdate cqn = addWhere(updateContext.getCqn(), wherePred);
					updateContext.setCqn(cqn);

				} else if (event.equals(DraftService.EVENT_DRAFT_EDIT)) {
					DraftEditEventContext draftEditContext = context.as(DraftEditEventContext.class);
					CqnSelect cqn = addWhere(draftEditContext.getCqn(), wherePred);
					draftEditContext.setCqn(cqn);

				} else if (event.equals(DraftService.EVENT_DRAFT_SAVE)) {
					DraftSaveEventContext saveContext = context.as(DraftSaveEventContext.class);
					CqnSelect cqn = addWhere(saveContext.getCqn(), wherePred);
					saveContext.setCqn(cqn);
				}

				if (logger.isDebugEnabled()) {
					logger.debug("Created CQN authorization predicate {} for entity '{}' of service '{}', event {} and user {}",
							wherePred.toJson(), entityName, serviceName, event, context.getUserInfo().getName());
				}

			} else if (logger.isDebugEnabled()) {
				logger.debug("No CQN authorization predicate required for entity '{}' of service '{}', event {} and user {}",
						entityName, serviceName, event, context.getUserInfo().getName());
			}
		} else if (regardCustomEvents && !CdsModelUtils.isStandardCdsEvent(event) && entityName != null) {
			CqnPredicate wherePred = authService.calcWhereCondition(entityName, event);
			if (wherePred != null) {
				if (rejectUnauthorized) {
					// check before WHERE condition is modified
					checkEntityAuthorization(context, wherePred);
				}
				if (DraftUtils.isDraftEnabled(targetEntity)) {
					// omit wherePred for draft entities. wherePred still applies for active entities.
					wherePred = CQL.or(wherePred, CQL.get(Drafts.IS_ACTIVE_ENTITY).eq(false));
				}
				CqnStatement cqn = (CqnStatement) context.get("cqn");
				if (cqn != null && (cqn.isSelect() || cqn.isUpdate() || cqn.isDelete())) {
					context.put("cqn", addWhere(cqn, wherePred));
				}
				if (logger.isDebugEnabled()) {
					logger.debug("Created CQN authorization predicate {} for entity '{}' of service '{}', event {} and user {}",
							wherePred.toJson(), entityName, serviceName, event, context.getUserInfo().getName());
				}
			}
		}
		// all checks derived from the model passes here.
		logger.debug("Instance-based authorization check passed for event '{}' on service '{}'", event, serviceName);
	}

	/**
	 * Reject UPDATE/DELETE/custom events if they target a single entity which is restricted and whose instance-based
	 * condition is not fulfilled.
	 *
	 * @todo Move to custom handler
	 *
	 * @param context UPDATE or DELETE event context.
	 * @param whereCondition `@restrict.where` value.
	 */
	private void checkEntityAuthorization(EventContext context, CqnPredicate whereCondition) {
		final String event = context.getEvent();
		if (!event.equals(CqnService.EVENT_DELETE) && !event.equals(CqnService.EVENT_UPDATE) &&
			!(regardCustomEvents && !CdsModelUtils.isStandardCdsEvent(event))) {
			return;
		}
		final CdsEntity targetEntity = context.getTarget();
		if (targetEntity == null) {
			return;
		}

		Map<String, Object> targetEntityKeys = getTargetEntityKeys(context);
		if (Boolean.FALSE.equals(targetEntityKeys.get(Drafts.IS_ACTIVE_ENTITY))) {
			// Since we don't handle draft events (yet), we know that the target is a draft entity,
			// hence no restrictions are applied, and we don't need to run the check.
			return;
		}

		Optional<CqnPredicate> keyFilter = keysToCondition(targetEntityKeys);
		if (keyFilter.isPresent()) {
			// Select those entities that match our keys and do _not_ pass the restriction.
			// If there are any rows that fail, we have to reject the whole filter.
			// `SELECT 1 as FAILED FROM Target WHERE (keyFilter) and NOT (whereCondition) LIMIT 1;`
			CqnSelect read = Select.from(targetEntity)
					.columns(c -> CQL.val(1).as("FAILED"))
					.where(CQL.and(keyFilter.get(), CQL.not(whereCondition)))
					.limit(1);

			long result = CdsServiceUtils.getDefaultPersistenceService(context).run(read).rowCount();
			if (result > 0L) {
				// To be 100% safe against information-disclosure, we'd need to check if the
				// user can read the entity and only then return 403.
				throw new ErrorStatusException(CdsErrorStatuses.EVENT_FORBIDDEN, event, targetEntity.getQualifiedName());
			}
		}
	}

	/**
	 * Extract target keys from the event's CQN.
	 * Returns an empty map if that can't be done, e.g. if keys are not unique or not complete.
	 */
	private Map<String, Object> getTargetEntityKeys(EventContext context) {
		CqnFilterableStatement cqn = (CqnFilterableStatement) context.get("cqn");
		if (cqn == null) {
			return Map.of();

		} else {
			CqnAnalyzer analyzer = CqnAnalyzer.create(context.getModel());
			AnalysisResult analysisResult = analyzer.analyze(cqn);
			return analysisResult.targetKeys();
		}
	}

	/**
	 * Transform the given keys into a CQN condition that matches them.  Returns an empty optional
	 * if not all keys are covered and the result wouldn't be unique.
	 */
	private Optional<CqnPredicate> keysToCondition(Map<String, Object> targetKeys) {
		if (targetKeys.isEmpty() || targetKeys.containsValue(null)) {
			// not all keys are set; can't uniquely identify targeted entity
			return Optional.empty();
		}
		CqnPredicate predicate = targetKeys.entrySet().stream()
				.map(entry -> CQL.get(entry.getKey()).eq(entry.getValue()))
				.collect(CQL.withAnd());
		return Optional.of(predicate);
	}

}
