/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.cds;

import java.util.Arrays;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.impl.parser.StructDataParser;
import com.sap.cds.impl.parser.StructDataParser.Include;
import com.sap.cds.reflect.CdsEvent;
import com.sap.cds.services.cds.RemoteService;
import com.sap.cds.services.environment.CdsProperties.Remote.RemoteServiceConfig;
import com.sap.cds.services.handler.Handler;
import com.sap.cds.services.impl.messaging.message.CdsMessagingUtils;
import com.sap.cds.services.messaging.CloudEventMessageEventContext;
import com.sap.cds.services.messaging.MessagingService;
import com.sap.cds.services.messaging.TopicMessageEventContext;
import com.sap.cds.services.messaging.utils.CloudEventUtils;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.StringUtils;

public class RemoteServiceImpl extends AbstractCdsDefinedService implements RemoteService {

	private static final Logger logger = LoggerFactory.getLogger(RemoteServiceImpl.class);

	private final Set<String> declaredEventsWithOnHandlers = ConcurrentHashMap.newKeySet();

	public RemoteServiceImpl(RemoteServiceConfig config, CdsRuntime runtime) {
		super(config.getName(), config.getModel(), runtime);
	}

	@Override
	public void on(String[] events, String[] entities, int order, Handler handler) {
		super.on(events, entities, order, handler);
		checkEvents(events);
	}

	private void checkEvents(String[] events) {
		MessagingService messagingService = CdsMessagingUtils.getMessagingService(runtime);
		if(messagingService != null) {
			Arrays.stream(events)
			.filter(event -> !StringUtils.isEmpty(event) && !event.equals("*"))
			.forEach(event -> {
				CdsEvent declaredEvent = getDefinition().events().filter(e -> e.getName().equals(event)).findFirst().orElse(null);
				if (declaredEvent != null && declaredEventsWithOnHandlers.add(declaredEvent.getQualifiedName())) {
					String topic = CdsMessagingUtils.getTopic(declaredEvent);
					logger.debug("Registering inbound event handler for declared event '{}' with topic '{}' on messaging service '{}' targeting remote service '{}'",
							declaredEvent.getQualifiedName(), topic, messagingService.getName(), getName());

					// register technical handler for declared event
					messagingService.on(topic, null, context -> {
						CloudEventMessageEventContext cloudContext = CloudEventUtils.toCloudEventMessageContext(context.as(TopicMessageEventContext.class), declaredEvent.getName());
						// validate CDS defined data structure
						mapDataTypes(declaredEvent, cloudContext);
						// emit the received message on the application service
						this.emit(cloudContext);
					});
				}
			});
		}
	}

	private void mapDataTypes(CdsEvent event, CloudEventMessageEventContext context) {
		try {
			// TODO: we have to serialize to JSON as the StructDataParser has no API for the map :(
			String jsonData = CloudEventUtils.toJson(context.getData());
			context.setData(StructDataParser.create(event).parseObject(jsonData, Include.GENERIC));
		} catch(Exception e) {
			throw new ErrorStatusException(CdsErrorStatuses.INVALID_DATA_FORMAT, event.getQualifiedName(), e);
		}
	}
}
