/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.draft;

import java.util.Collections;
import java.util.Iterator;
import java.util.Map;

import com.google.common.annotations.VisibleForTesting;
import com.sap.cds.ql.CQL;
import com.sap.cds.ql.cqn.AnalysisResult;
import com.sap.cds.ql.cqn.CqnAnalyzer;
import com.sap.cds.ql.cqn.CqnStatement;
import com.sap.cds.ql.cqn.ResolvedSegment;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.draft.Drafts;
import com.sap.cds.services.impl.utils.CdsModelUtils;
import com.sap.cds.services.utils.DraftUtils;

/**
 * Helper class to adapt draft queries for execution on the active and inactive persistence.
 */
public class CqnAdapter {

	private final EventContext context;
	private final CqnAnalyzer analyzer;

	private CqnAdapter(EventContext context) {
		this.context = context;
		this.analyzer = CqnAnalyzer.create(context.getModel());
	}

	public static CqnAdapter create(EventContext context) {
		return new CqnAdapter(context);
	}

	public <T extends CqnStatement> T adaptForActiveExecution(T stmt) {
		if (isValidForActiveExecution(stmt)) {
			return adapt(stmt, true);
		}
		return null;
	}

	public <T extends CqnStatement> T adaptForInactiveExecution(T stmt) {
		if (isValidForInactiveExecution(stmt)) {
			return adapt(stmt, false);
		}
		return null;
	}

	@VisibleForTesting
	boolean isValidForActiveExecution(CqnStatement statement) {
		AnalysisResult analysisResult = analyze(statement);
		Iterator<ResolvedSegment> iterator = analysisResult.iterator();
		while (iterator.hasNext()) {
			ResolvedSegment segment = iterator.next();
			if (DraftUtils.isDraftEnabled(segment.type())) {
				// no occurrence of IsActiveEntity = false allowed
				Boolean isActiveEntity = determineIsActiveEntity(segment, !iterator.hasNext() ? analysisResult.targetKeyValues() : Collections.emptyMap());
				if (Boolean.FALSE.equals(isActiveEntity)) {
					return false;
				}
			}
		}
		return true;
	}

	@VisibleForTesting
	boolean isValidForInactiveExecution(CqnStatement statement) {
		AnalysisResult analysisResult = analyze(statement);
		Iterator<ResolvedSegment> iterator = analysisResult.iterator();
		CdsEntity previous = null;
		while (iterator.hasNext()) {
			ResolvedSegment segment = iterator.next();
			boolean isDraftEnabled = DraftUtils.isDraftEnabled(segment.type());
			Boolean isActiveEntity = determineIsActiveEntity(segment, !iterator.hasNext() ? analysisResult.targetKeyValues() : Collections.emptyMap());
			if (previous == null) {
				// root must be draft-enabled and IsActiveEntity = true not allowed on root
				if (!isDraftEnabled || Boolean.TRUE.equals(isActiveEntity)) {
					return false;
				}
			} else if (isDraftEnabled) {
				// check the association target on the _drafts entity variant
				CdsEntity draftEntity = DraftModifier.getDraftsEntity(previous, context.getModel());
				boolean isSameDraftDocument = draftEntity.getTargetOf(segment.segment().id()).getName().endsWith(DraftModifier.DRAFT_SUFFIX);
				// when switching to IsActiveEntity = true
				// associations must not be followed, when staying in the draft document
				if (Boolean.TRUE.equals(isActiveEntity) && isSameDraftDocument) {
					return false;
				}
				// when switching to IsActiveEntity = false
				// associations must not be followed, when going outside the draft document
				if (Boolean.FALSE.equals(isActiveEntity) && !isSameDraftDocument) {
					return false;
				}
			}
			previous = segment.entity();
		}
		return true;
	}

	private <T extends CqnStatement> T adapt(T stmt, boolean isActiveExecution) {
		CdsEntity rootEntity = CdsModelUtils.getEntityPath(stmt, context.getModel()).rootEntity();
		boolean addSecurityConstraints = stmt.isSelect();
		boolean stripIsActiveEntity = !stmt.isInsert() && !stmt.isSelect();
		return CQL.copy(stmt, new DraftModifier(rootEntity, isActiveExecution, addSecurityConstraints, stripIsActiveEntity, context));
	}

	private AnalysisResult analyze(CqnStatement statement) {
		if (statement.isSelect()) {
			return analyzer.analyze(statement.asSelect());
		}
		if (statement.isUpdate()) {
			return analyzer.analyze(statement.asUpdate());
		}
		if (statement.isInsert()) {
			return analyzer.analyze(statement.asInsert().ref());
		}
		return analyzer.analyze(statement.asDelete());
	}

	private Boolean determineIsActiveEntity(ResolvedSegment segment, Map<String, Object> targetKeys) {
		Boolean result = (Boolean) segment.keyValues().get(Drafts.IS_ACTIVE_ENTITY);
		if (result == null && targetKeys.get(Drafts.IS_ACTIVE_ENTITY) instanceof Boolean) {
			result = (Boolean) targetKeys.get(Drafts.IS_ACTIVE_ENTITY);
		}
		return result;
	}

}
