/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.handlerregistry;

import java.lang.invoke.MethodHandle;
import java.util.ArrayList;
import java.util.List;

import com.sap.cds.services.handler.annotations.After;
import com.sap.cds.services.handler.annotations.Before;
import com.sap.cds.services.handler.annotations.HandlerOrder;
import com.sap.cds.services.handler.annotations.On;
import com.sap.cds.services.impl.handlerregistry.resolver.ArgumentResolver;
import com.sap.cds.services.impl.handlerregistry.resolver.EventContextArgumentResolver;
import com.sap.cds.services.impl.handlerregistry.resolver.PojoArgumentResolver;
import com.sap.cds.services.impl.handlerregistry.resolver.ReturnResolver;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;

/**
 * Meta descriptor of a event handler method.
 * This object stores information about the processed annotations, argument and return type resolvers and the order of a handler.
 */
public class HandlerDescriptor {

	private final String methodName;
	private final MethodHandle methodHandle;
	private final List<ArgumentResolver> argumentResolvers;
	private ReturnResolver returnResolver;

	private String[] defaultServiceNames = {};
	private Class<?>[] defaultServiceTypes = {};
	private int order = HandlerOrder.DEFAULT;
	private AnnotationDescriptor before;
	private AnnotationDescriptor on;
	private AnnotationDescriptor after;

	public HandlerDescriptor(String methodName, MethodHandle methodHandle, int parameterCount) {
		this.methodName = methodName;
		this.methodHandle = methodHandle;
		this.argumentResolvers = new ArrayList<>(parameterCount);
	}

	public String getMethodName() {
		return methodName;
	}

	public MethodHandle getMethodHandle() {
		return methodHandle;
	}

	public List<ArgumentResolver> getArgumentResolvers() {
		return this.argumentResolvers; // NOSONAR
	}

	public ReturnResolver getReturnResolver() {
		return this.returnResolver;
	}

	public void setReturnResolver(ReturnResolver returnResolver) {
		this.returnResolver = returnResolver;
	}

	public String[] getDefaultServiceNames() {
		return defaultServiceNames; // NOSONAR
	}

	public void setDefaultServiceNames(String[] defaultServiceNames) {
		this.defaultServiceNames = defaultServiceNames; // NOSONAR
	}

	public Class<?>[] getDefaultServiceTypes() {
		return defaultServiceTypes; // NOSONAR
	}

	public void setDefaultServiceTypes(Class<?>[] defaultServiceTypes) {
		this.defaultServiceTypes = defaultServiceTypes; // NOSONAR
	}

	public void setBefore(Before before) {
		this.before = new AnnotationDescriptor(before);
	}

	public void setOn(On on) {
		this.on = new AnnotationDescriptor(on);
	}

	public void setAfter(After after) {
		this.after = new AnnotationDescriptor(after);
	}

	public List<AnnotationDescriptor> getAnnotations() {
		List<AnnotationDescriptor> list = new ArrayList<>();
		if(this.before != null) {
			list.add(before);
		}
		if(this.on != null) {
			list.add(on);
		}
		if(this.after != null) {
			list.add(after);
		}
		return list;
	}

	public int getOrder() {
		return order;
	}

	public void setOrder(int order) {
		this.order = order;
	}

	public void verifyOrThrow() {
		int eventContextResolverCount = 0;
		int pojoResolverCount = 0;

		for(ArgumentResolver resolver : argumentResolvers) {
			resolver.verifyOrThrow(this);

			if(resolver instanceof EventContextArgumentResolver) {
				++eventContextResolverCount;
			}

			if(resolver instanceof PojoArgumentResolver) {
				++pojoResolverCount;
			}
		}

		if(eventContextResolverCount > 1) {
			throw new ErrorStatusException(CdsErrorStatuses.EVENT_CONTEXT_ARGUMENT_DUPLICATES, getMethodName());
		}

		if (pojoResolverCount > 1) {
			throw new ErrorStatusException(CdsErrorStatuses.POJO_ARGUMENT_DUPLICATES, getMethodName());
		}

		returnResolver.verifyOrThrow(this);
	}

}
