package com.sap.cds.services.impl.outbox.persistence;

import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicLong;

import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.runtime.CdsRuntimeAware;
import com.sap.cds.services.utils.info.CdsInfo;

public class CdsOutboxInfo implements CdsInfo, CdsRuntimeAware {
	private CdsRuntime runtime;

	@Override
	public void setCdsRuntime(CdsRuntime runtime) {
		this.runtime = runtime;
	}

	@Override
	public String name() {
		return "outboxes";
	}

	@Override
	public Object info(Details details) {
		Map<String, Object> result = new LinkedHashMap<>();
		List<PersistentOutbox> outboxServices = this.runtime
				.getServiceCatalog()
				.getServices(PersistentOutbox.class)
				.toList();

		if (!outboxServices.isEmpty()) {
			this.addOutboxNames(result, outboxServices);

			if (details.ordinal() >= Details.MEDIUM.ordinal()) {
				this.addOutboxTelemetryData(result, outboxServices);
			}
		} else {
			result.put("message", "No persistent outbox services configured.");
		}

		return result;
	}

	@SuppressWarnings("unchecked")
	private void addOutboxTelemetryData(Map<String, Object> result, List<PersistentOutbox> outboxServices) {
		outboxServices.forEach(outbox -> {
			Map<String, Object> outboxInfo = (Map<String, Object>) result.get(outbox.getName());
			Map<String, Long> outboxStatistics = (Map<String, Long>) outboxInfo.computeIfAbsent("statistics", k -> new HashMap<>());
			Collection<OutboxStatistics> statistics = outbox.getStatistics();

			AtomicLong statisticsCount = new AtomicLong();
			statistics.forEach(stats -> {
				statisticsCount.incrementAndGet();

				outboxStatistics.compute("incomingMessagesOfInstance", (k, v) -> v == null ? stats.getIncomingMessages() : v + stats.getIncomingMessages());
				outboxStatistics.compute("outgoingMessagesOfInstance", (k, v) -> v == null ? stats.getOutgoingMessages() : v + stats.getOutgoingMessages());

				outboxStatistics.compute("coldEntries", (k, v) -> v == null ? stats.getColdEntries() : v + stats.getColdEntries());
				outboxStatistics.compute("remainingEntries", (k, v) -> v == null ? stats.getRemainingEntries() : v + stats.getRemainingEntries());
				outboxStatistics.compute("maxStorageTime", (k, v) -> v == null ? stats.getMaxStorageTime() : Long.max(v, stats.getMaxStorageTime()));
				outboxStatistics.compute("avgStorageTime", (k, v) -> v == null ? stats.getMedianStorageTime() : v + stats.getMedianStorageTime());
				outboxStatistics.compute("minStorageTime", (k, v) -> v == null ? stats.getMinStorageTime() : Long.min(v, stats.getMinStorageTime()));
			});

			if (statisticsCount.get() > 1) {
				outboxStatistics.compute("avgStorageTime", (k, v) -> v == null ? 0 : v / statisticsCount.get());
			}
		});
	}

	private void addOutboxNames(Map<String, Object> result, List<PersistentOutbox> outboxServices) {
		outboxServices.forEach(outbox -> result.put(outbox.getName(), new HashMap<String, Object>()));
	}
}
