package com.sap.cds.services.impl.outbox.persistence;

import static com.sap.cds.services.impl.model.DynamicModelProvider.STATIC_MODEL_ACCESS_FEATURE;

import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Function;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.annotations.VisibleForTesting;
import com.sap.cds.Result;
import com.sap.cds.Row;
import com.sap.cds.ql.CQL;
import com.sap.cds.ql.Select;
import com.sap.cds.services.impl.outbox.Messages_;
import com.sap.cds.services.persistence.PersistenceService;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.OpenTelemetryUtils;

import io.opentelemetry.api.common.Attributes;
import io.opentelemetry.api.metrics.Meter;
import io.opentelemetry.api.metrics.ObservableLongMeasurement;

final class TelemetryDataImpl implements TelemetryData {

	private static final Logger logger = LoggerFactory.getLogger(TelemetryDataImpl.class);
	private static final Object NULL_TENANT = new Object();

	private record OutboxMetric(String name, Function<OutboxStatistics, Long> provider, String description, boolean isGauge) {};
	private static final String OUTBOX_INFO_INSTRUMENTATION_SCOPE = "com.sap.cds.outbox";
	private static final List<OutboxMetric> OUTBOX_METRICS = List.of(
			new OutboxMetric("com.sap.cds.outbox.coldEntries", OutboxStatistics::coldEntries, "Number of entries that could not be delivered after repeated attempts and will not be retried anymore.", true),
			new OutboxMetric("com.sap.cds.outbox.remainingEntries", OutboxStatistics::remainingEntries, "Number of entries which are pending for delivery.", true),
			new OutboxMetric("com.sap.cds.outbox.maxStorageTimeSeconds", OutboxStatistics::maxStorageTime, "Maximum time in seconds an entry was residing in the outbox.", true),
			new OutboxMetric("com.sap.cds.outbox.medStorageTimeSeconds", OutboxStatistics::medianStorageTime, "Median time in seconds of an entry stored in the outbox.", true),
			new OutboxMetric("com.sap.cds.outbox.minStorageTimeSeconds", OutboxStatistics::minStorageTime, "Minimal time in seconds an entry was stored in the outbox.", true),
			new OutboxMetric("com.sap.cds.outbox.incomingMessages", OutboxStatistics::incomingMessages, "Number of incoming messages of the outbox.", false),
			new OutboxMetric("com.sap.cds.outbox.outgoingMessages", OutboxStatistics::outgoingMessages, "Number of outgoing messages of the outbox.", false)
	);

	private final Map<Object, OutboxStatistics> statistics = new ConcurrentHashMap<>();
	private final List<ObservableLongMeasurement> observers = new ArrayList<>();

	private final String outboxName;
	private final int maxAttempts;

	public TelemetryDataImpl(String outboxName, int maxAttempts) {
		this.outboxName = outboxName;
		this.maxAttempts = maxAttempts;
		initializeOtel();
	}

	@Override
	public Collection<OutboxStatistics> getStatistics() {
		return this.statistics.values();
	}

	@VisibleForTesting
	OutboxStatistics getStatistics(String tenant) {
		return this.statistics.computeIfAbsent(tenant == null ? NULL_TENANT : tenant, t -> new OutboxStatistics(tenant));
	}

	/**
	 * Records the number of incoming messages for a given tenant and outbox.
	 *
	 * @param tenant     the tenant
	 * @param count      the number of incoming messages
	 */
	@Override
	public void recordIncomingMessages(String tenant, long count) {
		getStatistics(tenant).increaseIncomingMessages(count);
	}

	/**
	 * Records the number of outgoing messages for a given tenant and outbox.
	 *
	 * @param tenant     the tenant
	 * @param count      the number of outgoing messages
	 */
	@Override
	public void recordOutgoingMessages(String tenant, long count) {
		getStatistics(tenant).increaseOutgoingMessages(count);
	}

	/**
	 * Retrieves statistics for a given tenant and outbox, if the last update happened more than DELAY_IN_SECONDS seconds
	 * ago.This method expects to be called within an existing request context
	 *
	 * @param db the persistence service to use for retrieving the statistics
	 * @param tenant     the tenant
	 *
	 * @return true if the statistics were updated, false otherwise
	 */
	@Override
	public void recordStatistics(CdsRuntime runtime, PersistenceService db, String tenant) {
		logger.debug("Collecting statistics for outbox '{}' and tenant '{}'", this.outboxName, tenant);
		runtime.requestContext().featureToggles(STATIC_MODEL_ACCESS_FEATURE).systemUser(tenant).run(req -> {
			try {
				Select<?> selectCold = Select
						.from(Messages_.class)
						.columns(c -> CQL.count().as("count_cold"))
						.where(e ->
								e.target().eq(this.outboxName)
										.and(e.attempts().ge(this.maxAttempts)));
				Result coldCountResult = db.run(selectCold);
				long coldCount = ((Number) coldCountResult.single().get("count_cold")).longValue();

				Select<?> selectRemaining = Select
						.from(Messages_.class)
						.columns(
								c -> CQL.count().as("count_hot"),
								c -> CQL.min(c.timestamp()).as("maxTimestamp"),
								c -> CQL.max(c.timestamp()).as("minTimestamp"))
						.where(e ->
								e.target().eq(this.outboxName)
										.and(e.attempts().lt(maxAttempts)));
				Row remainingRow = db.run(selectRemaining).single();
				long remainingCount = ((Number) remainingRow.get("count_hot")).longValue();
				Instant maxTimestamp = (Instant) remainingRow.get("maxTimestamp");
				Instant minTimestamp = (Instant) remainingRow.get("minTimestamp");

				long medianIndex = remainingCount / 2L;
				Select<?> selectMedian = Select
						.from(Messages_.class)
						.columns(c -> c.timestamp().as("medTimestamp"))
						.where(e ->
								e.target().eq(this.outboxName)
										.and(e.attempts().lt(maxAttempts)))
						.orderBy(c -> c.timestamp().asc())
						.limit(1, medianIndex);
				Instant medianTimestamp = db.run(selectMedian).first().map(row -> (Instant) row.get("medTimestamp")).orElse(null);

				Instant now = Instant.now();
				long maxStorageTimeSeconds = maxTimestamp != null ? Duration.between(maxTimestamp, now).toSeconds() : 0;
				long medianStorageTimeSeconds = medianTimestamp != null ? Duration.between(medianTimestamp, now).toSeconds() : 0;
				long minStorageTimeSeconds = minTimestamp != null ? Duration.between(minTimestamp, now).toSeconds() : 0;

				OutboxStatistics stats = getStatistics(tenant);
				stats.setColdEntries(coldCount);
				stats.setRemainingEntries(remainingCount);
				stats.setMaxStorageTime(maxStorageTimeSeconds);
				stats.setMedianStorageTime(medianStorageTimeSeconds);
				stats.setMinStorageTime(minStorageTimeSeconds);

				logger.debug("Finished collecting outbox statistics for outbox '{}' and tenant '{}'", this.outboxName, tenant);
			} catch (Exception e) {
				logger.warn("Failed to collect statistics for outbox '{}' in tenant '{}'", this.outboxName, tenant, e);
			}
		});
	}

	private void initializeOtel() {
		// TODO try synchronous approach for gauges to avoid reporting outdated information
		Meter meter = OpenTelemetryUtils.getMeter(OUTBOX_INFO_INSTRUMENTATION_SCOPE);
		OUTBOX_METRICS.forEach(info -> {
			if (info.isGauge()) {
				observers.add(meter.gaugeBuilder(info.name())
						.setDescription(info.description()).ofLongs().buildObserver());
			} else {
				observers.add(meter.counterBuilder(info.name())
						.setDescription(info.description()).buildObserver());
			}
		});

		meter.batchCallback(
			this::recordAll,
			observers.get(0),
			observers.subList(1, observers.size()).toArray(new ObservableLongMeasurement[observers.size() - 1]));
	}

	private void recordAll() {
		logger.debug("Recording measurements for outbox '{}'", this.outboxName);
		List<String> measureLogs = new ArrayList<>(0);

		statistics.values().forEach(stats -> {
			Attributes attr = Attributes.of(OpenTelemetryUtils.CDS_TENANT, stats.getTenant(), OpenTelemetryUtils.CDS_OUTBOX_TARGET, this.outboxName);
			for (int i=0; i<observers.size(); ++i) {
				OutboxMetric metric = OUTBOX_METRICS.get(i);
				long value = metric.provider().apply(stats);
				if (value != -1) {
					observers.get(i).record(value, attr);
					if (logger.isTraceEnabled()) {
						measureLogs.add(String.format("%s(%s)=%d", metric.name(), stats.getTenant(), value));
					}
				}
			}
		});

		if (logger.isTraceEnabled()) {
			logger.trace("Recorded measurements for outbox '{}': {}", this.outboxName, String.join(", ", measureLogs));
		}
	}

}
