/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.runtime.mockusers;

import java.text.MessageFormat;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.TreeSet;

import com.sap.cds.services.environment.CdsProperties.Security.Mock.User;
import com.sap.cds.services.request.UserInfo;
import com.sap.cds.services.utils.ClassMethods;

public class MockedUserInfo implements UserInfo {

	private final User mockedUser;
	private final String userId;
	private final Set<String> roles;
	private final Map<String, List<String>> attributes;

	private static final String SPECIAL_ATTRIBUTE_TENANT = "tenant";

	public MockedUserInfo(User mockedUser) {
		this.mockedUser = mockedUser;
		this.userId = mockedUser.getId() == null ? ("mock/" + mockedUser.getName()) : mockedUser.getId();
		this.roles = new TreeSet<>(mockedUser.getRoles());
		this.attributes = new TreeMap<>(mockedUser.getAttributes());

		this.attributes.put(SPECIAL_ATTRIBUTE_TENANT, Collections.singletonList(mockedUser.getTenant()));
	}

	@Override
	public String getId() {
		return userId;
	}

	@Override
	public String getName() {
		return mockedUser.getName();
	}

	@Override
	public String getTenant() {
		return mockedUser.getTenant();
	}

	@Override
	public Set<String> getRoles() {
		return roles; // NOSONAR
	}

	@Override
	public boolean isSystemUser() {
		return mockedUser.isSystemUser() || isInternalUser(); // internal user implies system-user
	}

	@Override
	public boolean isInternalUser() {
		return mockedUser.isInternalUser();
	}
	
	@Override
	public boolean isAuthenticated() {
		return true; // otherwise this instance wouldn't have been created
	}

	@Override
	public boolean isPrivileged() {
		return mockedUser.isPrivileged();
	}

	@Override
	public Map<String, List<String>> getAttributes() {
		return attributes;
	}

	@Override
	public Map<String, Object> getAdditionalAttributes() {
		return mockedUser.getAdditional();
	}

	@Override
	public <T extends UserInfo> T as(Class<T> userInfoClazz) {
		return ClassMethods.as(userInfoClazz, UserInfo.class, this, this::getAdditionalAttributes);
	}

	@Override
	public String toString() {
		return MessageFormat.format("MockedUserInfo [id=''{0}'', name=''{1}'', roles=''{2}'', attributes=''{3}''", 
				getId(), getName(), getRoles(), getAttributes());
	}
}
