/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.aot;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.lang.reflect.Method;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.util.Iterator;

import org.graalvm.nativeimage.hosted.Feature;
import org.graalvm.nativeimage.hosted.RuntimeProxyCreation;
import org.graalvm.nativeimage.hosted.RuntimeReflection;

import com.sap.cds.CdsData;
import com.sap.cds.ql.StructuredType;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.Service;
import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.After;
import com.sap.cds.services.handler.annotations.Before;
import com.sap.cds.services.handler.annotations.On;
import com.sap.cds.services.impl.EventContextSPI;

public class CdsAotFeature implements Feature {

	@Override
	public void beforeAnalysis(BeforeAnalysisAccess access) {
		// EventContext proxies
		access.registerSubtypeReachabilityHandler((duringAccess, clazz) -> {
			// TODO avoid registering proxies for event contexts with concrete implementations
			if (clazz.isInterface() && !clazz.equals(EventContext.class) && !clazz.equals(EventContextSPI.class)) {
				RuntimeProxyCreation.register(clazz);
				RuntimeProxyCreation.register(clazz, EventContextSPI.class);
			}
		}, EventContext.class);

		// CdsData proxies
		access.registerSubtypeReachabilityHandler((duringAccess, clazz) -> {
			if (clazz.isInterface()) {
				RuntimeProxyCreation.register(clazz);
			}
		}, CdsData.class);

		// StructuredType proxies
		access.registerSubtypeReachabilityHandler((duringAccess, clazz) -> {
			if (clazz.isInterface() && !clazz.equals(StructuredType.class)) {
				RuntimeProxyCreation.register(clazz);
			}
		}, StructuredType.class);

		// Service proxies
		try {
			Iterator<URL> resources = ClassLoader.getSystemResources("META-INF/cds4j-codegen/services.generated").asIterator();
			while (resources.hasNext()) {
				try (InputStream generatedInput = resources.next().openStream()) {
					if (generatedInput != null) {
						try (BufferedReader reader = new BufferedReader(new InputStreamReader(generatedInput, StandardCharsets.UTF_8))) {
							reader.lines().forEach(this::registerServiceProxyForReflection);
						}
					}
				}
			}
		} catch (IOException e) {
			throw new RuntimeException("Could not load generated service interfaces", e);
		}

		// Event Handlers
		access.registerSubtypeReachabilityHandler((duringAccess, clazz) -> {
			registerEventHandlersForReflection(clazz);
		}, EventHandler.class);

		access.registerSubtypeReachabilityHandler((duringAccess, clazz) -> {
			registerEventHandlersForReflection(clazz);
			// sub-types of Service are potential outbox proxy candidates
			if (clazz.isInterface()) {
				RuntimeProxyCreation.register(clazz);
			}
		}, Service.class);
	}

	private void registerEventHandlersForReflection(Class<?> clazz) {
		for (Method m : clazz.getDeclaredMethods()) {
			if (m.isAnnotationPresent(Before.class) || m.isAnnotationPresent(On.class) || m.isAnnotationPresent(After.class)) {
				RuntimeReflection.register(m);
			} else {
				RuntimeReflection.registerAsQueried(m);
			}
		}
	}

	private void registerServiceProxyForReflection(String className) {
		try {
			ClassLoader loader = Thread.currentThread().getContextClassLoader();
			Class<?> clazz = loader.loadClass(className);
			RuntimeReflection.register(clazz);
			for (Method m : clazz.getDeclaredMethods()) {
				RuntimeReflection.register(m);
			}
			for (Class<?> proxyCandidate : clazz.getDeclaredClasses()) {
				if (proxyCandidate.isInterface()) {
					RuntimeReflection.register(proxyCandidate);
					RuntimeProxyCreation.register(proxyCandidate);
				}
			}
		} catch (ClassNotFoundException e) {
			throw new RuntimeException("Could not find generated service interface " + className, e);
		}
	}

}
