/**************************************************************************
 * (C) 2024-2025 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.authorization;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;
import java.util.Map;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Predicate;

import com.sap.cds.ql.cqn.CqnExistsSubquery;
import com.sap.cds.ql.cqn.CqnReference;
import com.sap.cds.ql.cqn.CqnReference.Segment;
import com.sap.cds.ql.cqn.ResolvedSegment;
import com.sap.cds.reflect.CdsAssociationType;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsStructuredType;

class ReferenceAssociationTraverser {

	static final Predicate<CdsElement> NO_COMPOSITIONS = e -> {
		CdsAssociationType type = e.getType().as(CdsAssociationType.class);
		return !type.isComposition() && !type.isEnclosed();
	};

	private ReferenceAssociationTraverser() {
		// hidden
	}

	/**
	 * Traverses the @{@link com.sap.cds.ql.cqn.CqnReference} represented as collection of segments and lets you
	 * replace the segment with a new one.
	 * As the last argument you may supply lambda to do something useful with the segment or its target
	 *
	 * @param root Structured type
	 * @param reference Reference to be modified
	 * @param takeIf boolean function to define if the association is considered
	 * @param action Some action
	 */
	static List<Segment> modify(CdsStructuredType root, CqnReference reference, Predicate<CdsElement> takeIf, Function<ResolvedSegment, Segment> action) {
		List<Segment> result = new ArrayList<>(reference.segments());

		ListIterator<Segment> iterator = result.listIterator();
		if (iterator.hasNext()) {
			ResolvedSegment next;
			// Absolute reference where the first segment has a type. Type is always visited
			if (reference.firstSegment().equals(root.getQualifiedName())) {
				next = new RefSegmentImpl(iterator.next(), root, null);
				iterator.set(action.apply(next));
			// Relative reference -> segment is used to carry the root type and visit rest of it segment by segment
			} else {
				next = new RefSegmentImpl(null, root, null);
			}

			while (iterator.hasNext()) {
				next = traverseSegment(next, iterator, takeIf, p -> iterator.set(action.apply(p)));
			}
		}
		return result;
	}

	/**
	 * Traverses the @{@link com.sap.cds.ql.cqn.CqnReference} represented as collection of segments and lets you
	 * observe them.
	 * As the last argument you may supply lambda to do something useful with the segment or its target
	 *
	 * @param root Structured type
	 * @param reference Reference to be modified
	 * @param takeIf boolean function to define if the association is considered
	 * @param action Some action
	 */
	static void consume(CdsStructuredType root, CqnReference reference, Predicate<CdsElement> takeIf, Consumer<ResolvedSegment> action) {
		Iterator<Segment> iterator = reference.segments().iterator();
		if (iterator.hasNext()) {
			ResolvedSegment next;
			// Absolute reference where the first segment has a type. Type is always visited
			if (reference.firstSegment().equals(root.getQualifiedName())) {
				next = new RefSegmentImpl(iterator.next(), root, null);
				action.accept(next);
			// Relative reference -> segment is used to carry the root type and visit rest of it segment by segment
			} else {
				next = new RefSegmentImpl(null, root, null);
			}

			while (iterator.hasNext()) {
				next = traverseSegment(next, iterator, takeIf, action);
			}
		}
	}

	private static ResolvedSegment traverseSegment(ResolvedSegment current, Iterator<Segment> iterator, Predicate<CdsElement> takeIf, Consumer<ResolvedSegment> action) {
		Segment segment = iterator.next();
		if (CqnExistsSubquery.OUTER.equals(segment.id())) {
			return current;
		}

		// REVISIT: On UCSN we can throw if element is absent and skip if it is a structure
		Optional<CdsElement> element = current.type().findElement(segment.id());
		if (element.filter(e -> e.getType().isAssociation() && takeIf.test(e)).isPresent()) {
			// It will be our next target and the navPath will be recreated as relative from new target
			current = new RefSegmentImpl(segment, element.get().getType().as(CdsAssociationType.class).getTarget(), element.get());
			action.accept(current);
		}
		return current;
	}

	private record RefSegmentImpl(Segment segment, CdsStructuredType type, CdsElement element) implements ResolvedSegment {
		// Just in case if Map.of() will no longer returns the same reference
		private static final Map<String, Object> EMPTY_MAP = Map.of();

		@Override
		public Map<String, Object> keys() {
			return EMPTY_MAP;
		}

		@Override
		public Map<String, Object> keyValues() {
			return EMPTY_MAP;
		}

		@Override
		public Map<String, Object> values() {
			return EMPTY_MAP;
		}
	}
}
