/*********************************************************************
 * (C) 2025 SAP SE or an SAP affiliate company. All rights reserved. *
 *********************************************************************/
package com.sap.cds.services.impl.cds;

import java.util.Map;

import com.sap.cds.Result;
import com.sap.cds.ql.CQL;
import com.sap.cds.ql.Select;
import com.sap.cds.ql.cqn.CqnAnalyzer;
import com.sap.cds.ql.cqn.CqnFilterableStatement;
import com.sap.cds.ql.cqn.CqnPredicate;
import com.sap.cds.ql.cqn.CqnSelect;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.draft.Drafts;
import com.sap.cds.services.impl.utils.CdsServiceUtils;
import com.sap.cds.services.utils.DraftUtils;
import com.sap.cds.util.DataUtils;

final class QueryAuthorizationCheckHelper {

	private QueryAuthorizationCheckHelper() {}

	static boolean checkAccessWithSelect(EventContext context, CqnPredicate keyFilter, CqnPredicate authFilter) {
		// Select those entities that match our keys and do _not_ pass the restriction.
		// If there are any rows that fail, we have to reject the whole filter.
		// `SELECT 1 as FAILED FROM Target WHERE (keyFilter) and NOT (whereCondition) LIMIT 1;`
		CqnSelect read = Select.from(context.getTarget())
			.columns(c -> CQL.val(1).as("FAILED"))
			.where(CQL.and(keyFilter, CQL.not(authFilter)))
			.limit(1);

		return CdsServiceUtils.getDefaultPersistenceService(context).run(read).rowCount() == 0;
	}

	static CqnPredicate keyFilterFromStatement(EventContext context) {
		if (context.get("cqn") instanceof CqnFilterableStatement statement) {
			CqnAnalyzer analyzer = CqnAnalyzer.create(context.getModel());
			Map<String, Object> targetKeys = analyzer.analyze(statement).targetKeys();

			if (hasMissingKeyValues(targetKeys)) {
				// Not all keys are set. We can't uniquely identify targeted entity.
				return CQL.FALSE;
			} else if (DraftUtils.isDraftEnabled(context.getTarget()) && Boolean.FALSE.equals(targetKeys.get(Drafts.IS_ACTIVE_ENTITY))) {
				// Since we don't handle draft events (yet), we know that the target is a draft entity,
				// hence no restrictions are applied, and we don't need to run the check.
				return CQL.FALSE;
			} else {
				return buildPredicate(targetKeys);
			}
		}
		return CQL.FALSE;
	}

	static CqnPredicate keyFilterFromResult(EventContext context, Result result) {
		// Result is not returned or result contains empty map (setters do that)
		if (result == null || result.first().filter(e -> !e.isEmpty()).isEmpty()) {
			return CQL.FALSE;
		}

		// Element names always follow flavour of CSN: they are either flattened or not
		// DataUtils lets us have normalized state where structured keys resolved for us
		// or null values when there is no keys in the payload.
		// This also strips IsActiveEntity that is also stripped during select normalization
		return result.stream().map(row -> {
			Map<String, Object> keyValues = DataUtils.keyValues(context.getTarget(), row);
			if (hasMissingKeyValues(keyValues)) {
				return CQL.FALSE;
			} else {
				return buildPredicate(keyValues);
			}
		}).collect(CQL.withOr());
	}

	private static boolean hasMissingKeyValues(Map<String, Object> targetKeys) {
		return targetKeys.isEmpty() || targetKeys.containsValue(null);
	}

	private static CqnPredicate buildPredicate(Map<String, Object> targetKeys) {
		return targetKeys.entrySet().stream()
			.map(entry -> CQL.get(entry.getKey()).eq(entry.getValue()))
			.collect(CQL.withAnd());
	}
}
