/*********************************************************************
 * (C) 2024 SAP SE or an SAP affiliate company. All rights reserved. *
 *********************************************************************/
package com.sap.cds.services.impl.cds;

import java.util.Iterator;

import com.sap.cds.ql.cqn.CqnExpand;
import com.sap.cds.ql.cqn.CqnInline;
import com.sap.cds.ql.cqn.CqnReference;
import com.sap.cds.ql.cqn.CqnSelectList;
import com.sap.cds.ql.cqn.CqnSelectListItem;
import com.sap.cds.ql.cqn.CqnVisitor;
import com.sap.cds.services.cds.ApplicationService;
import com.sap.cds.services.cds.CdsReadEventContext;
import com.sap.cds.services.cds.CqnService;
import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.Before;
import com.sap.cds.services.handler.annotations.HandlerOrder;
import com.sap.cds.services.handler.annotations.ServiceName;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.OrderConstants;

@ServiceName(value = "*", type = ApplicationService.class)
public class RejectExpandInlineAllHandler implements EventHandler {

	@Before(event = CqnService.EVENT_READ)
	@HandlerOrder(OrderConstants.Before.CHECK_STATEMENT)
	public void verifyStatement(CdsReadEventContext context) {
		StarChecker hasForbiddenStar = new StarChecker();
		context.getCqn().accept(hasForbiddenStar);
		if (hasForbiddenStar.isFound()) {
			throw new ErrorStatusException(CdsErrorStatuses.UNSUPPORTED_STAR);
		}
	}

	private static class StarChecker implements CqnVisitor {

		public boolean isFound() {
			return found;
		}

		private boolean found;

		@Override
		public void visit(CqnExpand expand) {
			checkSelectList(expand);
		}

		@Override
		public void visit(CqnInline inline) {
			checkSelectList(inline);
		}

		private void checkSelectList(CqnSelectList selector) {
			if (!found) {
				CqnReference ref = selector.ref();
				found = selector.ref().segments().isEmpty() || "*".equals(ref.targetSegment().id());

				if (!found) {
					visitItems(selector);
				}
			}
		}

		private void visitItems(CqnSelectList selector) {
			Iterator<CqnSelectListItem> iterator = selector.items().iterator();
			// Assumption: we need to check only inlines and expands, their refs at most
			// and go deeper in the items as needed.
			while(!found && iterator.hasNext()) {
				CqnSelectListItem i = iterator.next();
				if (i.isSelectList()) {
					i.accept(this);
				}
			}
		}

	}
}
