/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.messages;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.function.Predicate;
import java.util.stream.Stream;

import com.sap.cds.services.ErrorStatus;
import com.sap.cds.services.ErrorStatuses;
import com.sap.cds.services.ServiceException;
import com.sap.cds.services.messages.Message;
import com.sap.cds.services.messages.Message.Severity;
import com.sap.cds.services.messages.Messages;
import com.sap.cds.services.runtime.CdsRuntime;

public class MessagesImpl implements Messages {

	private ConcurrentLinkedQueue<Message> messages = new ConcurrentLinkedQueue<>();

	private final CdsRuntime runtime;
	private final Locale locale; // optional

	public MessagesImpl(CdsRuntime runtime, Locale locale) {
		this.runtime = runtime;
		this.locale = locale;
	}

	@Override
	public Message info(String messageOrKey, Object... args) {
		Message message = new MessageImpl(Severity.INFO, runtime.getLocalizedMessage(messageOrKey, args, locale));
		messages.add(message);
		return message;
	}

	@Override
	public Message success(String messageOrKey, Object... args) {
		Message message = new MessageImpl(Severity.SUCCESS, runtime.getLocalizedMessage(messageOrKey, args, locale));
		messages.add(message);
		return message;
	}

	@Override
	public Message warn(String messageOrKey, Object... args) {
		Message message = new MessageImpl(Severity.WARNING, runtime.getLocalizedMessage(messageOrKey, args, locale));
		messages.add(message);
		return message;
	}

	@Override
	public Message error(String messageOrKey, Object... args) {
		Message message = new MessageImpl(Severity.ERROR, runtime.getLocalizedMessage(messageOrKey, args, locale));
		messages.add(message);
		return message;
	}

	@Override
	public void throwIfError() throws ServiceException {
		messages.stream().filter(m -> m.getSeverity().equals(Severity.ERROR)).findFirst().ifPresent(error -> {
			messages.remove(error);
			ErrorStatus status = new ErrorStatus() {
				@Override
				public String getCodeString() {
					return error.getCode() != null ? error.getCode() : String.valueOf(getHttpStatus());
				}
				@Override
				public int getHttpStatus() {
					return ErrorStatuses.BAD_REQUEST.getHttpStatus();
				}
			};

			ServiceException exception = new ServiceException(status, error.getMessage())
				.messageTarget(error.getTarget())
				.longTextUrl(error.getLongTextUrl());
			throw exception;
		});
	}

	@Override
	public Stream<Message> stream() {
		return messages.stream();
	}

	@Override
	public List<Message> removeIf(Predicate<Message> filter) {
		List<Message> removed = new ArrayList<>();
		Iterator<Message> iterator = messages.iterator();
		while (iterator.hasNext()) {
			Message message = iterator.next();
			if (filter.test(message)) {
				iterator.remove();
				removed.add(message);
			}
		}
		return removed;
	}

}
