/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.outbox.persistence.collectors;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.services.mt.TenantProviderService;
import com.sap.cds.services.runtime.CdsRuntime;

/**
 * Coordinator implementation that manages all outbox partition collector
 * threads.
 */
public class TenantCache {

	private static final Logger LOG = LoggerFactory.getLogger(TenantCache.class);

	private final ScheduledExecutorService scheduler = Executors.newScheduledThreadPool(1);
	private TenantProviderService tenantService;
	private volatile List<String> tenantsInfo = new ArrayList<>();

	public List<String> getTenants() {
		return tenantsInfo;
	}

	public void start(CdsRuntime runtime) {
		if (this.tenantService == null) {
			this.tenantService = runtime.getServiceCatalog().getService(TenantProviderService.class, TenantProviderService.DEFAULT_NAME);

			long tenantsInfoRefreshIntervalSeconds = runtime.getEnvironment().getCdsProperties().getOutbox().getTenantsInfoRefreshInterval().getSeconds();
			this.scheduler.scheduleAtFixedRate(this::updateTenants, 0, tenantsInfoRefreshIntervalSeconds, TimeUnit.SECONDS);
			updateTenants();
		}
	}

	public void stop() {
		scheduler.shutdownNow();
	}

	private void updateTenants() {
		try {
			LOG.debug("Updating cached tenants for partition collectors");
			tenantsInfo = Collections.unmodifiableList(this.tenantService.readTenants());
		} catch (Exception e) {
			LOG.warn("Failed to update cached tenants for partition collectors", e);
		}
	}
}
