/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.docs.config;

import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

/**
 * Annotation is duplicated from
 * cds-configuration-processor/src/main/java/com/sap/cds/docs/config/DocumentedProperty.java
 * <p>
 * The reason is that we don't want to introduce a dependency on cds-configuration-processor.
 * We also don't have and don't want to introduce a shared module where we could move this file to.
 * Both of these options would require publishing an additional jar.
 * <p>
 * Reversing the dependency by moving it to cds-services-api and using Reflection to load the annotation dynamically
 * in cds-processing-processor isn't possible, because the classes aren't available yet when the mvn execution plugin
 * runs the PropertiesPostProcessor in cds-services-api.
 */
@Target({ElementType.TYPE, ElementType.FIELD})
@Retention(RetentionPolicy.RUNTIME)
public @interface DocumentedProperty {

	/**
	 * When set to {@code true} includes this property into the documentation.
	 * When set to {@code false} excludes this (otherwise transitively included) property from the documentation
	 *
	 * @return true, if the property should be included, false if it should be excluded from the documentation.
	 */
	boolean value() default true;

	/**
	 * When set to {@code false} excludes this nested property's header line from the documentation
	 *
	 * @return false, if the nested property's header line should be excluded
	 */
	boolean header() default true;

	/**
	 * When set to {@code true}, enforces traversing into this nested property class.
	 * By default only nested classes that are defined in processed class are considered.
	 * @return true, if traversing into this nested property class should be enforced.
	 */
	boolean forceNesting() default false;

	/**
	 * When set to {@code true} excludes the property's value from logging.
	 *
	 * @return true, if the property value may contain sensitive data, false otherwise.
	 */
	boolean sensitive() default false;

}
