/**************************************************************************
 * (C) 2019-2025 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.application;

import static java.lang.Math.max;

import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import javax.annotation.Nullable;

import org.apache.commons.lang3.ObjectUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.annotations.VisibleForTesting;
import com.sap.cds.docs.config.DocumentedProperty;
import com.sap.cds.services.application.ApplicationLifecycleService;
import com.sap.cds.services.application.ApplicationPreparedEventContext;
import com.sap.cds.services.environment.CdsProperties;
import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.On;
import com.sap.cds.services.handler.annotations.ServiceName;

/**
 * Handler to log cds properties on application prepared event.
 */
@ServiceName(value = "*", type = ApplicationLifecycleService.class)
public class ApplicationLifecycleLogCdsPropertiesHandler implements EventHandler {

    private final CdsPropertiesLogger cdsPropertiesLogger;
    private static final Logger logger = LoggerFactory.getLogger(ApplicationLifecycleLogCdsPropertiesHandler.class);

    public ApplicationLifecycleLogCdsPropertiesHandler() {
        this(new CdsPropertiesLogger());
    }

    @VisibleForTesting
    ApplicationLifecycleLogCdsPropertiesHandler(CdsPropertiesLogger logger) {
        this.cdsPropertiesLogger = logger;
    }

    /**
     * Logs messages when cds properties deviate from default values and warnings when the properties are deprecated
     * or not documented.
     * <br />
     * Per default, only warnings will be logged. To log all properties, set the log level to DEBUG using the
     * application property:
     * `logging.level.'com.sap.cds.properties'=DEBUG`
     * <br />
     * You can also turn it off to save on the resources during application startup by setting the log level to OFF.
     */
    @On
    public void logCdsProperties(ApplicationPreparedEventContext context) {
        if (!cdsPropertiesLogger.isLoggingEnabled()) {
            return;
        }

        var cdsProperties = context.getCdsRuntime().getEnvironment().getCdsProperties();
        logNonDefaults(cdsProperties);
    }

    private void logNonDefaults(CdsProperties actualValue) {
        var defaultValue = new CdsProperties();
        List<CdsPropertyLoggingContext> propertiesWithNonDefaultValues = identifyNonDefaultProperties(
                defaultValue, actualValue, "cds", false, true, false);
        cdsPropertiesLogger.log(propertiesWithNonDefaultValues);
    }

    /**
     * Recursive depth-first traversal of CdsProperties to identify properties with non-default values.
     *
     * @param defaultValue potentially null if called from list or map node
     * @param actualValue  potentially null if called from list or map node
     */
    private List<CdsPropertyLoggingContext> identifyNonDefaultProperties(@Nullable Object defaultValue,
                                                                         @Nullable Object actualValue,
                                                                         String parentPath,
                                                                         boolean isAncestorDeprecated,
                                                                         boolean isAncestorDocumented,
                                                                         boolean isAncestorSensitiveData) {
        List<CdsPropertyLoggingContext> propertiesWithNonDefaultValues = new ArrayList<>();

        if (needsUnwrapping(defaultValue, actualValue)) {
            propertiesWithNonDefaultValues.addAll(unwrapProperties(defaultValue, actualValue,
                    parentPath, isAncestorDeprecated, isAncestorDocumented, isAncestorSensitiveData));
        } else if (!Objects.equals(defaultValue, actualValue)) {
            propertiesWithNonDefaultValues.add(new CdsPropertyLoggingContext(defaultValue,
                    actualValue, parentPath, isAncestorDeprecated, isAncestorDocumented,
                    isAncestorSensitiveData));
        }

        return propertiesWithNonDefaultValues;
    }

    private List<CdsPropertyLoggingContext> unwrapProperties(@Nullable Object defaultValue,
                                                             @Nullable Object actualValue,
                                                             String parentPath, boolean isAncestorDeprecated,
                                                             boolean isAncestorDocumented,
                                                             boolean isAncestorSensitiveData) {
        List<CdsPropertyLoggingContext> propertiesWithNonDefaultValues = new ArrayList<>();

        if (defaultValue instanceof List || actualValue instanceof List) {
            var actualList = actualValue instanceof List<?> l ? l : Collections.emptyList();
            List<?> defaultList = defaultValue instanceof List<?> l ? l : Collections.emptyList();
            for (int i = 0; i < max(actualList.size(), defaultList.size()); i++) {
                // actual value missing from default list
                if (i < actualList.size()) {
                    var actualValue1 = actualList.get(i);
                    if (actualValue1 != null && !defaultList.contains(actualValue1)) {
                        propertiesWithNonDefaultValues.addAll(identifyNonDefaultProperties(null, actualValue1,
                                parentPath + "[" + i + "]", isAncestorDeprecated, isAncestorDocumented,
                                isAncestorSensitiveData));
                    }
                }

                // default value missing from actual list
                if (i < defaultList.size()) {
                    var defaultValue2 =  defaultList.get(i);
                    if (defaultValue2 != null && !actualList.contains(defaultValue2)) {
                        propertiesWithNonDefaultValues.addAll(identifyNonDefaultProperties(defaultValue2, null,
                                parentPath + "[" + i + "]", isAncestorDeprecated, isAncestorDocumented,
                                isAncestorSensitiveData));
                    }
                }
            }
        } else if (defaultValue instanceof Map || actualValue instanceof Map) {
            var actualMap = actualValue instanceof Map<?, ?> m ? m : Collections.emptyMap();
            var defaultMap = defaultValue instanceof Map<?, ?> m ? m : Collections.emptyMap();
            var unionKeySet = new HashSet<Object>(defaultMap.keySet());
            unionKeySet.addAll(actualMap.keySet());
            for (var key : unionKeySet) {
                var defaultValue1 = defaultMap.get(key);
                var actualValue1 = actualMap.get(key);
                propertiesWithNonDefaultValues.addAll(identifyNonDefaultProperties(defaultValue1, actualValue1,
                        parentPath + "." + key, isAncestorDeprecated, isAncestorDocumented,
                        isAncestorSensitiveData));
            }
        } else if (defaultValue != null && isCdsPropertyClass(defaultValue)
                || actualValue != null && isCdsPropertyClass(actualValue)) {
            propertiesWithNonDefaultValues.addAll(unwrapCdsPropertyClass(defaultValue, actualValue,
                    parentPath, isAncestorDeprecated, isAncestorDocumented, isAncestorSensitiveData));
        }
        return propertiesWithNonDefaultValues;
    }

    private List<CdsPropertyLoggingContext> unwrapCdsPropertyClass(@Nullable Object defaultValue,
                                                                   @Nullable Object actualValue,
                                                                   String parentPath, boolean isAncestorDeprecated,
                                                                   boolean isAncestorDocumented,
                                                                   boolean isAncestorSensitiveData) {
        List<CdsPropertyLoggingContext> propertiesWithNonDefaultValues = new ArrayList<>();
        if (defaultValue == null && actualValue == null) {
            return propertiesWithNonDefaultValues;
        }

        Object nonNullCdsProperties = ObjectUtils.firstNonNull(defaultValue, actualValue);
        var cdsPropertiesClass = nonNullCdsProperties.getClass();

        try {
            defaultValue = defaultValue != null ? defaultValue :
                    cdsPropertiesClass.getDeclaredConstructor().newInstance();
        } catch (InstantiationException | NoSuchMethodException | InvocationTargetException |
                 IllegalAccessException e) {
            // hopefully never reached because all classes should have default constructors for other reasons like
            // yaml deserialization as well
            logger.warn("Could not instantiate class '{}': {}", cdsPropertiesClass.getName(), e.getMessage());
            return propertiesWithNonDefaultValues;
        }

        for (var field : cdsPropertiesClass.getDeclaredFields()) {
            var getter = findGetter(field, cdsPropertiesClass);
            Object defaultFieldValue = getValue(defaultValue, field, getter, parentPath);
            Object actualFieldValue = getValue(actualValue, field, getter, parentPath);
            String propertyPath = parentPath + "." + field.getName();

            propertiesWithNonDefaultValues.addAll(identifyNonDefaultProperties(defaultFieldValue, actualFieldValue,
                    propertyPath, isDeprecated(field, isAncestorDeprecated), isDocumented(field, isAncestorDocumented),
                    isSensitiveData(field, isAncestorSensitiveData)));
        }
        return propertiesWithNonDefaultValues;
    }

    private Object getValue(@Nullable Object cdsProperties, Field field, Optional<Method> getter, String parentPath) {
        if (getter.isPresent() && cdsProperties != null) {
            try {
                return getter.get().invoke(cdsProperties);
            } catch (IllegalAccessException | InvocationTargetException e) {
                logger.warn("Exception while handling property '{}': {}", parentPath + field.getName(), e.getMessage());
            }
        }
        // expected for e.g. CdsProperties.MultiTenancy.SubscriptionManager.baseSecurity
        logger.trace("Could not find getter for field '{}.{}'.", parentPath, field.getName());
        return null;

    }

    private Optional<Method> findGetter(Field field, Class<?> cdsPropertiesClass) {
        String fieldName = field.getName().replace("_", "").toLowerCase(); // handle `_default`
        return Arrays.stream(cdsPropertiesClass.getDeclaredMethods()).filter(method -> {
            String methodName = method.getName().toLowerCase();
            return methodName.equals("get" + fieldName) || methodName.equals("is"
                    + fieldName);
        }).findFirst();
    }

    /**
     * Checks if the default defaultValue is a CDS object that needs unwrapping. This is the case for the internal
     * classes in CdsProperties, but should stop for all primitives, primitive wrappers and fun classes like
     * BigDecimal and Duration. If not for those, we could have used something like
     * ClassUtils.isPrimitiveOrWrapper(defaultValue.getClass())
     *
     * @implNote heavy assumption here that going forward all wrapped properties will still be internal classes.
     */
    private boolean needsUnwrapping(Object defaultValue, Object actualValue) {
        if (defaultValue != null) {
            return isCdsPropertyClass(defaultValue) || defaultValue instanceof List || defaultValue instanceof Map;
        } else if (actualValue != null) {
            return isCdsPropertyClass(actualValue) || actualValue instanceof List || actualValue instanceof Map;
        }
        return false;
    }

    private boolean isCdsPropertyClass(Object actualValue) {
        return actualValue.getClass().getName().startsWith(CdsProperties.class.getName());
    }

    /**
     * Check if the field or an ancestor is annotated with @DocumentedProperty(false)
     */
    private boolean isDocumented(Field field, boolean isAncestorDocumented) {
        var documentedProperty = field.getAnnotation(DocumentedProperty.class);
        return isAncestorDocumented && (documentedProperty == null || documentedProperty.value());
    }

    private boolean isDeprecated(Field field, boolean isAncestorDeprecated) {
        return isAncestorDeprecated || field.getAnnotation(Deprecated.class) != null;
    }


    /**
     * Check if the field or an ancestor is annotated with @DocumentedProperty(sensitive=true)
     */
    private boolean isSensitiveData(Field field, boolean isAncestorSensitiveData) {
        var documentedProperty = field.getAnnotation(DocumentedProperty.class);
        return isAncestorSensitiveData || (documentedProperty != null && documentedProperty.sensitive());
    }

}
