/**************************************************************************
 * (C) 2019-2025 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.application;


import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.List;

/**
 * Helper for ApplicationLifecycleLogCdsPropertiesHandler, so we can easily test the handler and mock this class instead
 * of having to test against the logging framework.
 * <p>
 * To turn on full logging, set the log level to DEBUG using the application property
 * `logging.level.properties=DEBUG`
 */
public class CdsPropertiesLogger {

    private static final Logger logger = LoggerFactory.getLogger(CdsPropertiesLogger.class);

    public boolean isLoggingEnabled() {
        return logger.isWarnEnabled();
    }

    /**
     * Logs messages for the passed cds properties. Debug level if they deviate from default values and warning when
     * the properties are deprecated or not documented.
     *
     * @param propertiesWithNonDefaultValues the list of cds properties which have non-default values
     */
    public void log(List<CdsPropertyLoggingContext> propertiesWithNonDefaultValues) {
        if (propertiesWithNonDefaultValues.isEmpty()) {
            logger.debug("All cds properties are set to default values.");
        }

        logProperties(propertiesWithNonDefaultValues);
    }

    /**
     * The early filter should be in sync with the conditions to adding warnings later in the method.
     */
    private void logProperties(List<CdsPropertyLoggingContext> propertiesWithNonDefaultValues) {
        propertiesWithNonDefaultValues.stream()
            .filter(property -> logger.isDebugEnabled() || property.isDeprecated() || !property.isDocumented())
            .forEach(propertyWithLoggingContext -> {
                var message = new StringBuilder();
                var logValue = propertyWithLoggingContext.containsSensitiveData() ? "***" :
                    propertyWithLoggingContext.actualValue();

                message.append("'").append(propertyWithLoggingContext.propertyPath()).append("': '").append(logValue).append("'");

                List<String> warnings = new ArrayList<>();
                if (propertyWithLoggingContext.isDeprecated()) {
                    warnings.add("deprecated");
                }
                if (!propertyWithLoggingContext.isDocumented()) {
                    warnings.add("not documented");
                }

                List<String> hints = new ArrayList<>();
                if (logDefaultValue(propertyWithLoggingContext)) {
                    hints.add(String.format("default: '%s'", propertyWithLoggingContext.defaultValue()));
                }
                hints.addAll(warnings);
                if (propertyWithLoggingContext.containsSensitiveData()) {
                    hints.add("sensitive");
                }

                if (!hints.isEmpty()) {
                    message.append(" (").append(String.join(", ", hints)).append(")");
                }

                if (!warnings.isEmpty()) {
                    logger.warn(message.toString());
                } else {
                    logger.debug(message.toString());
                }
            });
    }

    private boolean logDefaultValue(CdsPropertyLoggingContext propertyWithLoggingContext) {
        return propertyWithLoggingContext.defaultValue() != null && !propertyWithLoggingContext.defaultValue().equals("");
    }

}
