/**************************************************************************
 * (C) 2019-2025 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.cds;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.Result;
import com.sap.cds.ql.CQL;
import com.sap.cds.ql.cqn.CqnPredicate;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.authorization.AuthorizationService;
import com.sap.cds.services.cds.ApplicationService;
import com.sap.cds.services.cds.CqnService;
import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.After;
import com.sap.cds.services.handler.annotations.HandlerOrder;
import com.sap.cds.services.handler.annotations.ServiceName;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.OrderConstants;

@ServiceName(value = "*", type = ApplicationService.class)
public class InstanceBasedAuthorizationPayloadHandler implements EventHandler {

	private static final Logger logger = LoggerFactory.getLogger(InstanceBasedAuthorizationPayloadHandler.class);

	/**
	 * Reject CREATE/UPDATE events if they target a single entity which is restricted and whose
	 * instance-based condition is not fulfilled (for the payload).
	 */
	@After(event = {CqnService.EVENT_CREATE, CqnService.EVENT_UPDATE})
	@HandlerOrder(OrderConstants.After.CHECK_INSTANCE_BASED_PAYLOAD_AUTHORIZATION)
	private void checkPayloadAuthorization(EventContext context) {
		if (context.getUserInfo().isPrivileged()) {
			return;
		}

		CdsEntity target = context.getTarget();

		AuthorizationService authService = context.getServiceCatalog()
			.getService(AuthorizationService.class, AuthorizationService.DEFAULT_NAME);
		CqnPredicate authFilter = authService.calcWhereCondition(target.getQualifiedName(), context.getEvent());
		if (authFilter == null) {
			return; // no restriction => no check
		}

		// Entities without keys are not supported here
		if (context.getTarget().keyElements().findAny().isEmpty()) {
			logger.debug("Restriction on entity '{}' is ignored, because it has no keys", target.getQualifiedName());
			return;
		}

		// First step: if the statement references keys, we checked that update state still satisfies
		// the condition we already checked at @Before stage (see InstanceBasedAuthorizationRejectionHandler)
		CqnPredicate keyFilter = QueryAuthorizationCheckHelper.keyFilterFromStatement(context);
		if (keyFilter == CQL.FALSE) {
			// Or we try to derive similar filter using result of the statement,
			Result result = (Result) context.get("result");
			keyFilter = QueryAuthorizationCheckHelper.keyFilterFromResult(context, result);
			if (keyFilter == CQL.FALSE) { // null predicate or no predicate given -> no check done
				logger.debug("Payload of the entity defines no key values or the key values are incomplete. Restriction is ignored.");
				return;
			}
		}

		boolean hasAccess = QueryAuthorizationCheckHelper.checkAccessWithSelect(context, keyFilter, authFilter);
		if (!hasAccess) {
			throw new ErrorStatusException(CdsErrorStatuses.FORBIDDEN_VALUE, target.getQualifiedName());
		}
	}
}
