/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.draft;

import static java.util.stream.Collectors.toList;

import java.time.Instant;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.UUID;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.collect.ImmutableMap;
import com.sap.cds.CdsData;
import com.sap.cds.Result;
import com.sap.cds.Struct;
import com.sap.cds.impl.DataProcessor;
import com.sap.cds.ql.CQL;
import com.sap.cds.ql.Insert;
import com.sap.cds.ql.Select;
import com.sap.cds.ql.Update;
import com.sap.cds.ql.cqn.AnalysisResult;
import com.sap.cds.ql.cqn.CqnDelete;
import com.sap.cds.ql.cqn.CqnInsert;
import com.sap.cds.ql.cqn.CqnPredicate;
import com.sap.cds.ql.cqn.CqnSelect;
import com.sap.cds.ql.cqn.CqnStatement;
import com.sap.cds.ql.cqn.CqnStructuredTypeRef;
import com.sap.cds.ql.cqn.CqnUpdate;
import com.sap.cds.ql.cqn.ResolvedSegment;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsModel;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.draft.DraftAdministrativeData;
import com.sap.cds.services.draft.DraftCancelEventContext;
import com.sap.cds.services.draft.DraftCreateEventContext;
import com.sap.cds.services.draft.DraftNewEventContext;
import com.sap.cds.services.draft.DraftPatchEventContext;
import com.sap.cds.services.draft.DraftReadEventContext;
import com.sap.cds.services.draft.DraftService;
import com.sap.cds.services.draft.Drafts;
import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.After;
import com.sap.cds.services.handler.annotations.Before;
import com.sap.cds.services.handler.annotations.HandlerOrder;
import com.sap.cds.services.handler.annotations.On;
import com.sap.cds.services.handler.annotations.ServiceName;
import com.sap.cds.services.impl.draft.ParentEntityLookup.ParentEntityLookupResult;
import com.sap.cds.services.impl.utils.CdsModelUtils;
import com.sap.cds.services.impl.utils.CdsServiceUtils;
import com.sap.cds.services.persistence.PersistenceService;
import com.sap.cds.services.request.RequestContext;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.DraftUtils;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.OrderConstants;
import com.sap.cds.services.utils.TenantAwareCache;
import com.sap.cds.services.utils.model.CqnUtils;
import com.sap.cds.util.OnConditionAnalyzer;

/**
 * Event handlers for events on inactive versions of draft-enabled entities.
 */
@ServiceName(value = "*", type = DraftService.class)
public class InactiveHandler implements EventHandler {

	private static final Logger logger = LoggerFactory.getLogger(InactiveHandler.class);
	private final TenantAwareCache<ParentEntityLookup, CdsModel> parentEntityLookups;

	public InactiveHandler(CdsRuntime runtime) {
		parentEntityLookups = TenantAwareCache.create(() -> new ParentEntityLookup(RequestContext.getCurrent(runtime).getModel()), runtime);
	}

	// TODO additional capabilities checks for additional READ events similar to CapabilitiesHandler?
	@Before(event = {DraftService.EVENT_DRAFT_NEW, DraftService.EVENT_DRAFT_PATCH, DraftService.EVENT_DRAFT_CANCEL})
	@HandlerOrder(OrderConstants.Before.CHECK_CAPABILITIES)
	protected void checkDraft(EventContext context) {
		if (!DraftUtils.isDraftEnabled(context.getTarget())) {
			throw new ErrorStatusException(CdsErrorStatuses.ENTITY_NOT_DRAFT_ENABLED, context.getTarget().getQualifiedName());
		}
	}

	@On
	@HandlerOrder(OrderConstants.On.DEFAULT_ON)
	protected Result defaultRead(DraftReadEventContext context) {
		CqnSelect adapted = CqnAdapter.create(context).adaptForInactiveExecution(context.getCqn());
		if (adapted == null) {
			return DraftHandlerUtils.buildNoOpResult(context);
		}
		return CdsServiceUtils.getDefaultPersistenceService(context).run(adapted, context.getCqnNamedValues());
	}

	@On
	@HandlerOrder(OrderConstants.On.DEFAULT_ON)
	protected Result defaultNew(DraftNewEventContext context) {
		return DraftServiceImpl.downcast(context.getService()).createDraft(Insert.copy(context.getCqn()), false);
	}

	@On
	@HandlerOrder(OrderConstants.On.DEFAULT_ON)
	protected Result defaultCreate(DraftCreateEventContext context) {
		CqnInsert adapted = CqnAdapter.create(context).adaptForInactiveExecution(context.getCqn());
		if (adapted == null) {
			return DraftHandlerUtils.buildNoOpResult(context);
		}
		return CdsServiceUtils.getDefaultPersistenceService(context).run(adapted);
	}

	@On
	@HandlerOrder(OrderConstants.On.DEFAULT_ON)
	protected Result defaultPatch(DraftPatchEventContext context) {
		CqnUpdate adapted = CqnAdapter.create(context).adaptForInactiveExecution(context.getCqn());
		if (adapted == null) {
			return DraftHandlerUtils.buildNoOpResult(context);
		}
		adapted = addSecurityConstraints(adapted, context);
		return CdsServiceUtils.getDefaultPersistenceService(context).run(adapted, context.getCqnValueSets());
	}

	@On
	@HandlerOrder(OrderConstants.On.DEFAULT_ON)
	protected Result defaultCancel(DraftCancelEventContext context) {
		CqnDelete adapted = CqnAdapter.create(context).adaptForInactiveExecution(context.getCqn());
		if (adapted == null) {
			return DraftHandlerUtils.buildNoOpResult(context);
		}
		adapted = addSecurityConstraints(adapted, context);
		return CdsServiceUtils.getDefaultPersistenceService(context).run(adapted, context.getCqnValueSets());
	}

	@After
	protected void validateDraftParent(DraftNewEventContext context) {
		if (!DraftUtils.isDraftRoot(context.getTarget())) {
			// check if the entity has a parent entity in draft mode
			List<ParentEntityLookupResult> parentLookups = parentEntityLookups.findOrCreate().lookupParent(context.getTarget());
			if (parentLookups == null || parentLookups.isEmpty()) {
				throw new ErrorStatusException(CdsErrorStatuses.NO_PARENT_ENTITY, context.getTarget().getQualifiedName());
			}
			context.getResult().forEach(r -> {
				int numberParents = 0;
				for (ParentEntityLookupResult parentLookup: parentLookups) {
					if (DraftUtils.isDraftEnabled(parentLookup.getParentEntity())) {
						String compositionName = parentLookup.getComposition().getName();
						CqnPredicate pred = CQL.get(Drafts.IS_ACTIVE_ENTITY).eq(false);
						for (Map.Entry<String, Object> key: getKeys(r, context.getTarget()).entrySet()) {
							CqnPredicate tmpPred = CQL.to(compositionName).get(key.getKey()).eq(key.getValue());
							pred = CQL.and(pred, tmpPred);
						}
						CqnSelect select = Select.from(parentLookup.getParentEntity()).where(pred);
						Result parentDrafts = DraftServiceImpl.downcast(context.getService()).readDraft(select);
						if (parentDrafts.first().isPresent()) {
							++numberParents;
						}
					}
				}
				if (numberParents == 0) {
					throw new ErrorStatusException(CdsErrorStatuses.PARENT_NOT_EXISTING, context.getTarget().getQualifiedName());
				}
				if (numberParents > 1) {
					throw new ErrorStatusException(CdsErrorStatuses.MULTIPLE_PARENTS, numberParents, context.getTarget().getQualifiedName());
				}
			});
		}
	}

	@After
	protected void updateAdminDataOnNewDraft(DraftNewEventContext context) {
		if (!DraftUtils.isDraftRoot(context.getTarget())) {
			updateDraftAdministrativeDataChangeTime(context.getResult(), context);
		}
	}

	@Before
	@SuppressWarnings("unchecked")
	@HandlerOrder(HandlerOrder.AFTER + OrderConstants.Before.FILTER_FIELDS)
	protected void updateAdminDataOnPatch(DraftPatchEventContext context) {
		List<Map<String, Object>> entries = context.getCqn().entries();
		// TODO should we be able to get the NOW time stamp from the context?
		Instant now = Instant.now();
		DataProcessor.create().forUpdate().action((entity, row) -> {
			if (entity.findElement(Drafts.DRAFT_ADMINISTRATIVE_DATA).isPresent()) {
				if (!entity.getQualifiedName().equals(context.getTarget().getQualifiedName())) {
					// deep updates on draft entities are not supported
					throw new ErrorStatusException(CdsErrorStatuses.DRAFT_DEEP_UPDATE);
				}
				DraftAdministrativeData adminData;
				if (row.containsKey(Drafts.DRAFT_ADMINISTRATIVE_DATA)) {
					adminData = Struct.access((Map<String, Object>) row.get(Drafts.DRAFT_ADMINISTRATIVE_DATA))
							.as(DraftAdministrativeData.class);
				} else {
					adminData = Struct.create(DraftAdministrativeData.class);
					row.put(Drafts.DRAFT_ADMINISTRATIVE_DATA, adminData);
				}
				adminData.setLastChangeDateTime(now);
			}
		}).process(entries, context.getTarget());
	}

	@Before
	@HandlerOrder(OrderConstants.Before.CALCULATE_FIELDS)
	protected void updateAdminDataOnDeletion(DraftCancelEventContext context) {
		if (!DraftUtils.isDraftRoot(context.getTarget())) {
			Select<?> select = CqnUtils.toSelect(context.getCqn()).columns(Drafts.DRAFT_ADMINISTRATIVE_DATA);
			updateDraftAdministrativeDataChangeTime(DraftServiceImpl.downcast(context.getService()).readDraft(select), context);
		}
	}

	@Before
	@HandlerOrder(HandlerOrder.AFTER + OrderConstants.Before.FILTER_FIELDS)
	protected void addDraftFields(DraftCreateEventContext context, List<CdsData> entries) {
		// TODO should we be able to get the NOW time stamp from the context?
		Instant now = Instant.now();
		for (CdsData m : entries) {
			final String draftUUID = determineDraftUuid(m, context);
			DataProcessor.create().forInsert().action((entity, data) -> {
				if (DraftUtils.isDraftEnabled(entity)) {
					Drafts draft = Struct.access(data).as(Drafts.class);
					draft.setHasActiveEntity(context.getHasActiveEntity());
					draft.setHasDraftEntity(false);
					draft.setIsActiveEntity(false);
					draft.setDraftAdministrativeDataDraftUuid(draftUUID);

					DraftAdministrativeData dad = draft.getDraftAdministrativeData();
					if (dad == null) {
						dad = Struct.create(DraftAdministrativeData.class);
					}
					dad.setDraftUUID(draftUUID);
					if (DraftUtils.isDraftRoot(entity)) {
						String userName = context.getUserInfo().getName();
						dad.setCreatedByUser(userName);
						dad.setCreationDateTime(now);
						dad.setLastChangeDateTime(now);
						dad.setInProcessByUser(userName);
						dad.setLastChangedByUser(userName);
					}
					draft.setDraftAdministrativeData(dad);
				}
			}).process(m, context.getTarget());
		}
	}

	private <S extends CqnStatement> S addSecurityConstraints(S stmt, EventContext context) {
		return CqnUtils.addWhere(stmt, DraftModifier.getSecurityConstraints(context));
	}

	private String determineDraftUuid(Map<String, Object> m, DraftCreateEventContext context) {
		Drafts draftData = Struct.access(m).as(Drafts.class);
		if (draftData.getDraftAdministrativeDataDraftUuid() != null) {
			// for forward linked associations we cannot determine the draftUUID from the data
			// if the foreign key is not set in the parent entity, therefore it must be provided
			return draftData.getDraftAdministrativeDataDraftUuid();
		} else if (draftData.getDraftAdministrativeData() != null && draftData.getDraftAdministrativeData().getDraftUUID() != null) {
			return draftData.getDraftAdministrativeData().getDraftUUID();
		} else if (DraftUtils.isDraftRoot(context.getTarget())) {
			// root entity -> we generate DraftAdministrativeData
			return UUID.randomUUID().toString();
		}
		// child entity -> we need to find out the parent entity to retrieve the draftUUID
		Result result = null;
		long numParents = 0;
		Set<String> uniqueParents = new HashSet<>();
		CqnStructuredTypeRef ref = context.getCqn().ref();
		List<ParentEntityLookupResult> lookupResultList = parentEntityLookups.findOrCreate().lookupParent(context.getTarget());
		AnalysisResult analysisResult = CdsModelUtils.getEntityPath(ref, context.getModel());
		Iterator<ResolvedSegment> iter = analysisResult.reverse();
		iter.next();
		CdsEntity potentialParent = iter.hasNext() ? iter.next().entity() : null;
		if (potentialParent != null && lookupResultList.stream().anyMatch(lr -> lr.getParentEntity().equals(potentialParent))) {
			// we have a path expression where the parent entity comes before the last segment -> we get the parent by omitting the last segment
			CqnSelect select = Select.from(CQL.to(ref.segments().subList(0, ref.size() - 1)))
					.columns(Drafts.DRAFT_ADMINISTRATIVE_DATA);
			result = DraftServiceImpl.downcast(context.getService()).readDraft(select);
			numParents = result.rowCount();
		} else {
			// the child is the root entity of the statement -> we need to find the parent through a lookup
			Result tmpResult;
			for (ParentEntityLookupResult lookupResult: lookupResultList) {
				CdsEntity parentEntity = lookupResult.getParentEntity();

				if (!uniqueParents.contains(parentEntity.getQualifiedName()) && DraftUtils.isDraftEnabled(parentEntity)) {
					uniqueParents.add(parentEntity.getQualifiedName());

					OnConditionAnalyzer analyzer = new OnConditionAnalyzer(lookupResult.getComposition(), false);

					// m is the whole entry map of the child insert, but
					// analyzer.getFkValues(m) only returns the foreign key of the child that refers to the parent
					Map<String, Object> foreignKeysParent = analyzer.getFkValues(m);
					foreignKeysParent.put(Drafts.IS_ACTIVE_ENTITY, false);

					// therefore, the select here only selects the parent's "DraftAdministrativeData_DraftUUID" column
					// from the parent with its ID as specified as the FK in the child
					CqnSelect select = Select.from(parentEntity).matching(foreignKeysParent).columns(Drafts.DRAFT_ADMINISTRATIVE_DATA);
					tmpResult = DraftServiceImpl.downcast(context.getService()).readDraft(select);

					// if 'lookupResultList' contains more than one entry due to several unmanaged compositions defined in the parent's model
					// the parent entity in all entries refers to the same, therefore the query would produce a result for each 'lookupResultList'
					// entry and thus 'numParents' would be more than 1 if we had not checked for unique parents before entering the if clause
					// TODO this is only correct if they share the same backlink, therefore the backlink association or plain fk fields need to be considered here as well for uniqueness
					// TODO this needs to prioritize structured foreign keys
					numParents += tmpResult.rowCount();
					if (tmpResult.first().isPresent()) {
						result = tmpResult;
					}
					if (numParents > 1) {
						break;
					}
				}
			}
		}

		if (numParents > 1) {
			throw new ErrorStatusException(CdsErrorStatuses.MULTIPLE_PARENTS, numParents, context.getTarget().getQualifiedName());
		}
		if (numParents == 0 || result == null) {
			throw new ErrorStatusException(CdsErrorStatuses.PARENT_NOT_EXISTING, context.getTarget().getQualifiedName());
		}
		DraftAdministrativeData data = result.single().as(Drafts.class).getDraftAdministrativeData();
		return data != null ? data.getDraftUUID() : null;
	}

	private static Map<String, Object> getKeys(Map<String, Object> m, CdsEntity entity) {
		Map<String, Object> result = new HashMap<>(m);
		result.entrySet().removeIf(entry -> {
			return !entity.findElement(entry.getKey()).map(e -> e.isKey()).orElse(false)
					&& !Drafts.IS_ACTIVE_ENTITY.equals((entry.getKey()));

		});
		if (result.size() != entity.keyElements().filter(element -> !element.getType().isAssociation()).count()) {
			logger.error("Failed to find keys of {} in result {}", entity, m);
			throw new ErrorStatusException(CdsErrorStatuses.NO_KEYS_IN_RESULT, entity);
		}
		return result;
	}

	private void updateDraftAdministrativeDataChangeTime(Result result, EventContext context) {
		Instant now = Instant.now();
		List<Map<String, Object>> updates = result.streamOf(Drafts.class)
				.map(row -> row.getDraftAdministrativeData()).filter(Objects::nonNull)
				.map(row -> (Object) row.getDraftUUID()).filter(Objects::nonNull)
				.distinct()
				.map(uuid -> ImmutableMap.of(DraftAdministrativeData.DRAFT_UUID, uuid, DraftAdministrativeData.LAST_CHANGE_DATE_TIME, now))
				.collect(toList());
		if (!updates.isEmpty()) {
			PersistenceService persistenceService = context.getCdsRuntime().getServiceCatalog().getService(PersistenceService.class, PersistenceService.DEFAULT_NAME);
			persistenceService.run(Update.entity(DraftAdministrativeData.CDS_NAME).entries(updates));
		}
	}

}
