package com.sap.cds.services.impl.draft.messages;

import static com.sap.cds.services.impl.draft.messages.DraftMessageUtils.MESSAGES_ELEMENT;
import static com.sap.cds.services.impl.draft.messages.DraftMessageUtils.MESSAGES_PATH;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import com.sap.cds.CdsData;
import com.sap.cds.Struct;
import com.sap.cds.ql.CQL;
import com.sap.cds.ql.cqn.CqnElementRef;
import com.sap.cds.ql.cqn.CqnStructuredTypeRef;
import com.sap.cds.ql.cqn.CqnValue;
import com.sap.cds.ql.cqn.Modifier;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.draft.DraftReadEventContext;
import com.sap.cds.services.draft.DraftService;
import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.After;
import com.sap.cds.services.handler.annotations.Before;
import com.sap.cds.services.handler.annotations.HandlerOrder;
import com.sap.cds.services.handler.annotations.ServiceName;
import com.sap.cds.services.messages.MessageLookup;
import com.sap.cds.services.messages.MessageTarget;
import com.sap.cds.services.utils.DraftUtils;
import com.sap.cds.services.utils.ODataUtils;
import com.sap.cds.services.utils.OrderConstants;

@ServiceName(value = "*", type = DraftService.class)
class DraftMessagesReadHandler implements EventHandler {

	@Before
	@HandlerOrder(OrderConstants.Before.ADAPT_STATEMENT)
	void selectMessages(DraftReadEventContext context) {
		if (!DraftUtils.isDraftEnabled(context.getTarget())) {
			return;
		}

		context.setCqn(CQL.copy(context.getCqn(), new Modifier() {

			@Override
			public CqnValue ref(CqnElementRef ref) {
				if (ref.path().equals(MESSAGES_ELEMENT)) {
					return CQL.get(MESSAGES_PATH);
				}
				return ref;
			}

		}));
	}

	@After
	@HandlerOrder(OrderConstants.After.ADD_FIELDS)
	@SuppressWarnings("unchecked")
	void postProcessMessages(DraftReadEventContext context, List<CdsData> dataList) {
		if (!DraftUtils.isDraftEnabled(context.getTarget())) {
			return;
		}

		// only return messages of current node and its children
		CqnStructuredTypeRef refForTarget = DraftMessageUtils.forTarget(context.getCqn().ref());
		String requestTarget = refForTarget != null ? ODataUtils.toODataTarget(null, null, refForTarget, false) : null;

		for (CdsData data : dataList) {
			var draftMessages = (Collection<Map<String, Object>>) data.get(MESSAGES_ELEMENT);
			if (draftMessages != null) {
				Iterator<Map<String, Object>> iterator = draftMessages.iterator();
				while (iterator.hasNext()) {
					DraftMessage draftMessage = Struct.access(iterator.next()).as(DraftMessage.class);
					Targets targets = relevantTargets(requestTarget, draftMessage, context);
					if (requestTarget == null || targets.target != null) {
						MessageLookup lookup = DraftMessageUtils.parseMessageLookup(draftMessage.getMessage());
						String localized = context.getCdsRuntime().getLocalizedMessage(lookup.getMessageOrKey(), lookup.getArgs(), context.getParameterInfo().getLocale());
						draftMessage.setMessage(localized);
						if (targets.target != null) {
							draftMessage.setTarget(targets.target);
						}
						if (!targets.additionalTargets.isEmpty()) {
							draftMessage.setAdditionalTargets(targets.additionalTargets);
						}
					} else {
						iterator.remove();
					}
				}
			}
		}
	}

	private record Targets(String target, List<String> additionalTargets) {};

	private Targets relevantTargets(String requestTarget, DraftMessage draftMessage, EventContext context) {
		String target = getEffectiveTarget(requestTarget, draftMessage.getTarget());
		List<String> additionalTargets = List.of();
		if (draftMessage.getAdditionalTargets() != null) {
			additionalTargets = new ArrayList<>(draftMessage.getAdditionalTargets().size());
			for (String additionalTarget : draftMessage.getAdditionalTargets()) {
				String effectiveAdditionalTarget = getEffectiveTarget(requestTarget, additionalTarget);
				if (effectiveAdditionalTarget != null) {
					additionalTargets.add(effectiveAdditionalTarget);
				}
			}
		}
		if (target != null) {
			return new Targets(target, additionalTargets);
		}
		return new Targets(!additionalTargets.isEmpty() ? additionalTargets.remove(0) : null, additionalTargets);
	}

	private String getEffectiveTarget(String requestTarget, String rawTarget) {
		MessageTarget target = DraftMessageUtils.parseMessageTarget(rawTarget);
		if (target != null) {
			String odataTarget = ODataUtils.toODataTarget(null, target.getParameter(), target.getRef(), false);
			if (requestTarget == null || odataTarget != null && (odataTarget.equals(requestTarget) || odataTarget.startsWith(requestTarget + "/"))) {
				if (requestTarget != null) {
					odataTarget = odataTarget.substring(requestTarget.length() + (odataTarget.equals(requestTarget) ? 0 : 1));
				}
				return odataTarget;
			}
		}
		return null;
	}

}
