package com.sap.cds.services.impl.draft.messages;

import static com.sap.cds.services.draft.DraftService.EVENT_DRAFT_CANCEL;
import static com.sap.cds.services.draft.DraftService.EVENT_DRAFT_NEW;
import static com.sap.cds.services.draft.DraftService.EVENT_DRAFT_PATCH;
import static com.sap.cds.services.draft.DraftService.EVENT_DRAFT_SAVE;
import static com.sap.cds.services.impl.draft.messages.DraftMessageUtils.MESSAGES_ELEMENT;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.sap.cds.CdsData;
import com.sap.cds.CdsDataProcessor;
import com.sap.cds.Result;
import com.sap.cds.Struct;
import com.sap.cds.ql.CQL;
import com.sap.cds.ql.RefBuilder;
import com.sap.cds.ql.Select;
import com.sap.cds.ql.StructuredTypeRef;
import com.sap.cds.ql.Update;
import com.sap.cds.ql.cqn.CqnAnalyzer;
import com.sap.cds.ql.cqn.CqnReference.Segment;
import com.sap.cds.ql.cqn.CqnSelect;
import com.sap.cds.ql.cqn.CqnStatement;
import com.sap.cds.ql.cqn.CqnStructuredTypeRef;
import com.sap.cds.ql.cqn.CqnUpdate;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.draft.DraftAdministrativeData;
import com.sap.cds.services.draft.DraftCreateEventContext;
import com.sap.cds.services.draft.DraftService;
import com.sap.cds.services.draft.Drafts;
import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.After;
import com.sap.cds.services.handler.annotations.Before;
import com.sap.cds.services.handler.annotations.HandlerOrder;
import com.sap.cds.services.handler.annotations.ServiceName;
import com.sap.cds.services.impl.cds.ValidationErrorHandler;
import com.sap.cds.services.impl.draft.DraftServiceImpl;
import com.sap.cds.services.messages.Message;
import com.sap.cds.services.messages.MessageTarget;
import com.sap.cds.services.utils.DraftUtils;
import com.sap.cds.services.utils.ODataUtils;
import com.sap.cds.services.utils.OrderConstants;

@ServiceName(value = "*", type = DraftService.class)
class DraftMessagesWriteHandler implements EventHandler {

	public static final String DRAFT_MESSAGES_SKIP_HINT = "cds.draftMessages.skip";
	private static final Set<String> DRAFT_VALIDATION_EVENTS = Set.of(EVENT_DRAFT_NEW, EVENT_DRAFT_PATCH, EVENT_DRAFT_CANCEL);
	private static final String STORE_MESSAGES = "cds.internal.draftMessages.storeMessages";

	@Before
	@HandlerOrder(HandlerOrder.AFTER + OrderConstants.Before.FILTER_FIELDS)
	void isStorageRequired(EventContext context, CqnStatement cqn) {
		if (!(DRAFT_VALIDATION_EVENTS.contains(context.getEvent()) || isActionOnDraft(context, cqn))
			|| !DraftUtils.isDraftEnabled(context.getTarget())
			|| cqn.hints().containsKey(DRAFT_MESSAGES_SKIP_HINT)
			|| !isById(cqn.ref(), context)) {
			return;
		}
		context.put(STORE_MESSAGES, true);
		context.put(ValidationErrorHandler.THROW_IF_ERROR_IN_AFTER, true);
	}

	@Before
	@HandlerOrder(OrderConstants.Before.CALCULATE_FIELDS)
	void initializeMessages(DraftCreateEventContext context, List<CdsData> dataList) {
		if (DraftUtils.isDraftRoot(context.getTarget())) {
			for (CdsData data : dataList) {
				DraftAdministrativeData dad = Struct.access(data).as(Drafts.class).getDraftAdministrativeData();
				if (dad != null && dad.get(MESSAGES_ELEMENT) == null) {
					dad.put(MESSAGES_ELEMENT, new ArrayList<>(0));
				}
			}
		}
	}

	@Before
	@HandlerOrder(OrderConstants.Before.THROW_IF_ERROR)
	@SuppressWarnings("unchecked")
	void throwOnInvalidActionParameters(EventContext context) {
		if (!Boolean.TRUE.equals(context.get(STORE_MESSAGES))) {
			return; // not messages relevant
		}

		// still throw in @Before if action parameters are invalid
		// @On handlers can still rely on validated action parameters
		Set<Message> previousMessages = (Set<Message>) context.get(ValidationErrorHandler.PREVIOUS_MESSAGES_KEY);
		if (context.getMessages().stream().anyMatch(m -> !previousMessages.contains(m)
				&& m.isTransition() && m.getSeverity().equals(Message.Severity.ERROR))) {
			context.getMessages().throwIfError();
		}
	}

	@After
	@HandlerOrder(OrderConstants.After.THROW_IF_ERROR - 1)
	@SuppressWarnings("unchecked")
	void persistMessages(EventContext context, CqnStructuredTypeRef ref, Result result) {
		if (!Boolean.TRUE.equals(context.get(STORE_MESSAGES))) {
			return; // not messages relevant
		}

		String event = context.getEvent();
		List<String> patchedTargets = new ArrayList<>();
		if (result != null && result.rowCount() == 1) {
			// collect patched targets
			if (EVENT_DRAFT_PATCH.equals(event)) {
				final CqnStructuredTypeRef requestRef = ref;
				CdsDataProcessor.create().addValidator((p, e, t) -> true, (p, e, v) -> {
					patchedTargets.add(DraftMessageUtils.stringifyMessageTarget(
						DraftMessageUtils.prefixMessageTarget(requestRef, MessageTarget.create(p, e))));
				}).process(result, context.getTarget());
			}

			// get target filter segment with keys from result ref for NEW
			if (EVENT_DRAFT_NEW.equals(event) && ref.targetSegment().filter().isEmpty()) {
				RefBuilder<StructuredTypeRef> copy = CQL.copy(ref);
				result.single().ref().asRef().targetSegment().filter().ifPresent(f -> copy.targetSegment().filter(f));
				ref = copy.build();
			}
		}

		Set<Message> previousMessages = (Set<Message>) context.get(ValidationErrorHandler.PREVIOUS_MESSAGES_KEY);
		List<Message> newMessages = context.getMessages().removeIf(m -> !previousMessages.contains(m) && !m.isTransition());
		// we don't need to store messages, in case the draft root is deleted or if no messages have been created or might be invalidated
		if ((!EVENT_DRAFT_CANCEL.equals(event) || ref.size() > 1) && (!newMessages.isEmpty() || requiresInvalidation(patchedTargets, event, ref))) {
			List<DraftMessage> draftMessages = loadMessages(context, ref);
			boolean invalidated = invalidateMessages(draftMessages, patchedTargets, ref, context);
			for (Message message : newMessages) {
				draftMessages.add(DraftMessageUtils.toDraftMessage(message, ref));
			}
			if (invalidated || !newMessages.isEmpty()) {
				CqnStructuredTypeRef updateRef = EVENT_DRAFT_CANCEL.equals(event) ? parentOf(ref) : ref;
				Map<String, Object> data = new HashMap<>(Map.of(Drafts.DRAFT_ADMINISTRATIVE_DATA, Map.of(MESSAGES_ELEMENT, draftMessages)));
				data.putAll(CqnAnalyzer.create(context.getModel()).analyze(updateRef).targetKeyValues()); // TODO remove once adjusted in CDS4j
				CqnUpdate writeDraftMessages = Update.entity(updateRef).data(data).hint(DRAFT_MESSAGES_SKIP_HINT, true);
				((DraftService) context.getService()).patchDraft(writeDraftMessages);
			}
		}
	}

	@SuppressWarnings("unchecked")
	private List<DraftMessage> loadMessages(EventContext context, CqnStructuredTypeRef ref) {
		String event = context.getEvent();
		List<DraftMessage> messages = new ArrayList<>();
		// no need to load messages, in case the draft root is created
		if (!EVENT_DRAFT_NEW.equals(event) || ref.size() > 1) {
			// load all state messages
			CqnSelect queryDraftMessages = Select.from(toDraftAdministrativeData(EVENT_DRAFT_NEW.equals(event) || EVENT_DRAFT_CANCEL.equals(event) ? parentOf(ref) : ref));
			Result draftMessagesResult = DraftServiceImpl.downcast(context.getService()).readDraft(queryDraftMessages);
			if (draftMessagesResult.rowCount() == 1) {
				Collection<Map<String, Object>> draftMessages = (Collection<Map<String, Object>>) draftMessagesResult.single().get(MESSAGES_ELEMENT);
				if (draftMessages != null) {
					Struct.stream(draftMessages).as(DraftMessage.class).forEach(messages::add);
				}
			}
		}
		return messages;
	}

	private boolean requiresInvalidation(List<String> patchedTargets, String event, CqnStructuredTypeRef ref) {
		return (EVENT_DRAFT_PATCH.equals(event) && !patchedTargets.isEmpty()) || (EVENT_DRAFT_CANCEL.equals(event) && ref.size() > 1);
	}

	private boolean invalidateMessages(List<DraftMessage> draftMessages, List<String> patchedTargets, CqnStructuredTypeRef ref, EventContext context) {
		String event = context.getEvent();
		if (EVENT_DRAFT_PATCH.equals(event) && !patchedTargets.isEmpty()) {
			return draftMessages.removeIf(draftMessage ->
				patchedTargets.stream().anyMatch(patchedTarget ->
					patchedTarget.equals(draftMessage.getTarget()) ||
					draftMessage.getAdditionalTargets() != null && draftMessage.getAdditionalTargets().stream().anyMatch(t -> patchedTarget.equals(t))
				)
			);
		} else if (EVENT_DRAFT_CANCEL.equals(event) && ref.size() > 1) {
			// invalidate this target and all its children
			CqnStructuredTypeRef refForTarget = DraftMessageUtils.forTarget(ref);
			String deletedTarget = ODataUtils.toODataTarget(null, null, refForTarget, false);
			return draftMessages.removeIf(m ->
				matchesDeletedTarget(deletedTarget, m.getTarget()) ||
				m.getAdditionalTargets() != null && m.getAdditionalTargets().stream().anyMatch(t -> matchesDeletedTarget(deletedTarget, t))
			);
		}
		return false;
	}

	private boolean matchesDeletedTarget(String deletedTarget, String stringifiedTarget) {
		MessageTarget target = DraftMessageUtils.parseMessageTarget(stringifiedTarget);
		if (target != null) {
			String odataTarget = ODataUtils.toODataTarget(null, target.getParameter(), target.getRef(), false);
			return deletedTarget.equals(odataTarget) || odataTarget.startsWith(deletedTarget + "/");
		}
		return false;
	}

	private static boolean isActionOnDraft(EventContext context, CqnStatement cqn) {
		if (context.getTarget() != null && !EVENT_DRAFT_SAVE.equals(context.getEvent()) &&
				context.getTarget().findAction(context.getEvent()).isPresent()) {
			Object isActiveEntity = CqnAnalyzer.create(context.getModel()).analyze(cqn.ref()).targetKeyValues().get(Drafts.IS_ACTIVE_ENTITY);
			return Boolean.FALSE.equals(isActiveEntity);
		}
		return false;
	}

	private static boolean isById(CqnStructuredTypeRef ref, EventContext context) {
		return ref.targetSegment().filter().isPresent() || EVENT_DRAFT_NEW.equals(context.getEvent());
	}

	private static CqnStructuredTypeRef parentOf(CqnStructuredTypeRef ref) {
		return CQL.to(ref.segments().subList(0, ref.size() - 1)).asRef();
	}

	private static CqnStructuredTypeRef toDraftAdministrativeData(CqnStructuredTypeRef ref) {
		List<Segment> segments = new ArrayList<>(ref.segments());
		segments.add(CQL.refSegment(Drafts.DRAFT_ADMINISTRATIVE_DATA));
		return CQL.to(segments).asRef();
	}

}
