/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.outbox;

import java.util.HashSet;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.services.environment.CdsProperties.Outbox;
import com.sap.cds.services.environment.CdsProperties.Outbox.OutboxServiceConfig;
import com.sap.cds.services.impl.outbox.persistence.PersistentOutbox;
import com.sap.cds.services.impl.outbox.persistence.PersistentOutboxInitializationHandler;
import com.sap.cds.services.impl.outbox.persistence.collectors.TenantCache;
import com.sap.cds.services.impl.scheduler.TaskScheduler;
import com.sap.cds.services.outbox.OutboxService;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.runtime.CdsRuntimeConfiguration;
import com.sap.cds.services.runtime.CdsRuntimeConfigurer;
import com.sap.cds.services.utils.outbox.OutboxUtils;

public class OutboxServiceConfiguration implements CdsRuntimeConfiguration {

	private static final Logger logger = LoggerFactory.getLogger(OutboxServiceConfiguration.class);

	private final TenantCache tenantCache = new TenantCache();
	private TaskScheduler taskScheduler;

	@Override
	public void services(CdsRuntimeConfigurer configurer) {
		CdsRuntime runtime = configurer.getCdsRuntime();
		Outbox outboxConfig = runtime.getEnvironment().getCdsProperties().getOutbox();

		if (outboxConfig.getPersistent().isEnabled()) {
			if (OutboxUtils.hasOutboxModel(runtime.getCdsModel())) {
				taskScheduler = null;
				if (runtime.getEnvironment().getCdsProperties().getTaskScheduler().isEnabled()) {
					taskScheduler = new TaskScheduler(runtime);
				}

				Set<String> disabled = new HashSet<>();
				outboxConfig.getServices().values().forEach(config -> {
					if (config.isEnabled()) {
						configurer.service(new PersistentOutbox(config.getName(), overwriteOrderedFlag(config), runtime, taskScheduler, tenantCache::getTenants));
					} else {
						disabled.add(config.getName());
					}
				});

				if (runtime.getServiceCatalog().getService(OutboxService.class, OutboxService.PERSISTENT_ORDERED_NAME) == null
						&& !disabled.contains(OutboxService.PERSISTENT_ORDERED_NAME)) {

					OutboxServiceConfig defaultConfig = overwriteOrderedFlag(new OutboxServiceConfig(OutboxService.PERSISTENT_ORDERED_NAME));
					configurer.service(new PersistentOutbox(OutboxService.PERSISTENT_ORDERED_NAME, defaultConfig, runtime, taskScheduler, tenantCache::getTenants));
				}

				if (runtime.getServiceCatalog().getService(OutboxService.class, OutboxService.PERSISTENT_UNORDERED_NAME) == null
						&& !disabled.contains(OutboxService.PERSISTENT_UNORDERED_NAME)) {

					OutboxServiceConfig defaultConfig = overwriteOrderedFlag(new OutboxServiceConfig(OutboxService.PERSISTENT_UNORDERED_NAME));
					configurer.service(new PersistentOutbox(OutboxService.PERSISTENT_UNORDERED_NAME, defaultConfig, runtime, taskScheduler, tenantCache::getTenants));
				}
			} else if (!outboxConfig.getServices().isEmpty()) {
				logger.warn("Explicitly configured outbox services will not be available, because persistent outboxes model is not available.");
			}
		}

		if (outboxConfig.getInMemory().isEnabled()) {
			configurer.service(new InMemoryOutbox(OutboxService.INMEMORY_NAME, runtime));
		}
	}

	private OutboxServiceConfig overwriteOrderedFlag(OutboxServiceConfig config) {
		if (OutboxService.PERSISTENT_ORDERED_NAME.equals(config.getName())) {
			config.setOrdered(true);
		} else if (OutboxService.PERSISTENT_UNORDERED_NAME.equals(config.getName())) {
			config.setOrdered(false);
		}

		return config;
	}

	@Override
	public void eventHandlers(CdsRuntimeConfigurer configurer) {
		if (configurer.getCdsRuntime().getServiceCatalog().getServices(PersistentOutbox.class).findAny().isPresent()) {
			configurer.eventHandler(new PersistentOutboxInitializationHandler(tenantCache, taskScheduler));
		}
		configurer.eventHandler(new CqnServiceOutboxHandler());
	}

	@Override
	public int order() {
		return -100; // before usual service configurations
	}

}
