/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.outbox.persistence;

import java.time.Duration;
import java.util.List;
import java.util.concurrent.ThreadLocalRandom;

import com.sap.cds.services.application.ApplicationLifecycleService;
import com.sap.cds.services.application.ApplicationPreparedEventContext;
import com.sap.cds.services.application.ApplicationStoppedEventContext;
import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.After;
import com.sap.cds.services.handler.annotations.On;
import com.sap.cds.services.handler.annotations.ServiceName;
import com.sap.cds.services.impl.outbox.persistence.collectors.TenantCache;
import com.sap.cds.services.impl.scheduler.TaskScheduler;
import com.sap.cds.services.impl.scheduler.TaskScheduler.Task;
import com.sap.cds.services.impl.scheduler.TaskScheduler.TaskSchedule;
import com.sap.cds.services.mt.TenantProviderService;
import com.sap.cds.services.runtime.CdsRuntime;

@ServiceName(ApplicationLifecycleService.DEFAULT_NAME)
public class PersistentOutboxInitializationHandler implements EventHandler {

	private final TenantCache tenantCache;
	private final TaskScheduler taskScheduler;

	public PersistentOutboxInitializationHandler(TenantCache tenantCache, TaskScheduler taskScheduler) {
		this.tenantCache = tenantCache;
		this.taskScheduler = taskScheduler;
	}

	@After
	protected void initOutboxCollectors(ApplicationPreparedEventContext context) {
		if (context.getCdsRuntime().getEnvironment().getCdsProperties().getEnvironment().getCommand().isEnabled()) {
			return;
		}

		if (taskScheduler != null) {
			taskScheduler.start();
		} else {
			tenantCache.start(context.getCdsRuntime());
		}

		context.getCdsRuntime().getServiceCatalog().getServices(PersistentOutbox.class).forEach(PersistentOutbox::init);

		if (taskScheduler != null) {
			taskScheduler.scheduleTask("OutboxTenantsTask", new AllTenantsTask(context.getCdsRuntime()), Duration.ofSeconds(10).toMillis());
		}
	}

	@On
	protected void stopOutboxCollectors(ApplicationStoppedEventContext context) {
		context.getCdsRuntime().getServiceCatalog().getServices(PersistentOutbox.class).forEach(PersistentOutbox::stop);
		if (taskScheduler != null) {
			taskScheduler.shutdown();
		} else {
			tenantCache.stop();
		}
	}


	/**
	 * All tenants task is responsible for periodic planning of tenant task for all outboxes
	 */
	private static class AllTenantsTask implements Task {

		private static final long ALL_TENANTS_INTERVALS = Duration.ofHours(1).toMillis();
		private static final long RANDOM_INTERVALS = Duration.ofMinutes(10).toMillis();

		private final TenantProviderService tenantService;
		private final List<PersistentOutbox> outboxServices;

		public AllTenantsTask(CdsRuntime runtime) {
			this.tenantService = runtime.getServiceCatalog().getService(TenantProviderService.class, TenantProviderService.DEFAULT_NAME);
			this.outboxServices = runtime.getServiceCatalog().getServices(PersistentOutbox.class).toList();
		}

		@Override
		public TaskSchedule run() {
			tenantService.readTenants().forEach(tenant -> {
				outboxServices.forEach(outbox -> {
					long delay = ThreadLocalRandom.current().nextLong(RANDOM_INTERVALS);
					outbox.schedule(tenant, delay, true);
				});
			});
			return new TaskSchedule(this, ALL_TENANTS_INTERVALS, true);
		}
	}
}
