/**************************************************************************
 * (C) 2019-2025 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.outbox.persistence.collectors;

import static com.sap.cds.services.impl.model.DynamicModelProvider.STATIC_MODEL_ACCESS_FEATURE;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.time.Instant;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ThreadLocalRandom;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.core.Version;
import com.fasterxml.jackson.core.util.VersionUtil;
import com.sap.cds.CdsLockTimeoutException;
import com.sap.cds.Result;
import com.sap.cds.Struct;
import com.sap.cds.impl.parser.JsonParser;
import com.sap.cds.ql.CQL;
import com.sap.cds.ql.Delete;
import com.sap.cds.ql.Predicate;
import com.sap.cds.ql.Select;
import com.sap.cds.ql.Update;
import com.sap.cds.ql.cqn.CqnSelect;
import com.sap.cds.services.environment.CdsProperties.Outbox.OutboxServiceConfig;
import com.sap.cds.services.impl.outbox.Messages;
import com.sap.cds.services.impl.outbox.Messages_;
import com.sap.cds.services.impl.outbox.persistence.PersistentOutbox;
import com.sap.cds.services.impl.outbox.persistence.TelemetryData;
import com.sap.cds.services.impl.scheduler.TaskScheduler;
import com.sap.cds.services.impl.scheduler.TaskScheduler.Task;
import com.sap.cds.services.impl.scheduler.TaskScheduler.TaskSchedule;
import com.sap.cds.services.outbox.OutboxMessage;
import com.sap.cds.services.outbox.OutboxMessageEventContext;
import com.sap.cds.services.persistence.PersistenceService;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.OpenTelemetryUtils;
import com.sap.cds.services.utils.lib.mt.TenantUtils;

import io.opentelemetry.api.trace.Span;
import io.opentelemetry.api.trace.SpanKind;
import io.opentelemetry.context.Scope;

/**
 * Outbox collector implementation based on {@link TaskScheduler}.
 */
public class TaskBasedCollector implements OutboxCollector {

	private static final Logger LOG = LoggerFactory.getLogger(TaskBasedCollector.class);

	private final CdsRuntime runtime;
	private final PersistentOutbox outboxService;
	private final TaskScheduler taskScheduler;
	private final TelemetryData telemetryData;
	private final String target;

	private final boolean storeLastError;
	private final int maxPublishAttempts;
	private final boolean ordered;

	private final long maxFailPause;
	private final boolean checkVersion;
	private final String appVersion;
	private final Version appVersionParsed;
	private final Set<String> suspendedTenants = ConcurrentHashMap.newKeySet();

	private PersistenceService db;
	private boolean isStarted;

	public TaskBasedCollector(CdsRuntime runtime, PersistentOutbox outboxService, OutboxServiceConfig config, String appVersion, TaskScheduler taskScheduler, TelemetryData telemetryData) {
		this.runtime = runtime;
		this.outboxService = outboxService;
		this.taskScheduler = taskScheduler;
		this.telemetryData = telemetryData;
		this.target = outboxService.getName();

		this.storeLastError = config.getStoreLastError().isEnabled();
		this.maxPublishAttempts = config.getMaxAttempts();
		this.ordered = config.isOrdered();

		this.maxFailPause = config.getMaxPause().toMillis();
		this.checkVersion = config.isCheckVersion();
		this.appVersion = appVersion;
		this.appVersionParsed = checkVersion ? VersionUtil.parseVersion(appVersion, null, null) : null;

		if (!storeLastError) {
			LOG.debug("Storing errors for outbox '{}' is disabled.", outboxService.getName());
		}
	}

	@Override
	public void start() {
		LOG.debug("Starting collector of the outbox '{}'", outboxService.getName());
		this.db = runtime.getServiceCatalog().getService(PersistenceService.class, PersistenceService.DEFAULT_NAME);
		isStarted = true;
	}

	@Override
	public void stop(long waitMillis) {
		LOG.debug("Stopping collector of the outbox '{}'", outboxService.getName());
		isStarted = false;
	}

	@Override
	public boolean isRunning() {
		return isStarted;
	}

	@Override
	public void schedule(String tenant, long delay, boolean withEmptyCheck) {
		if (!suspendedTenants.contains(tenant)) {
			taskScheduler.scheduleTask(target + "/" + tenant, new CollectorTask(tenant, withEmptyCheck), delay);
		}
	}

	private class CollectorTask implements Task {

		private final String tenant;
		private final boolean withEmptyCheck;

		public CollectorTask(String tenant, boolean withEmptyCheck) {
			this.tenant = tenant;
			this.withEmptyCheck = withEmptyCheck;
		}

		@Override
		public boolean isReady() {
			return isStarted;
		}

		@Override
		public TaskSchedule run() {
			if (suspendedTenants.contains(tenant)) {
				// outdated instance we should not plan any tasks for this tenant
				return null;
			}

			Optional<Span> span = OpenTelemetryUtils.createSpan(OpenTelemetryUtils.CdsSpanType.OUTBOX, SpanKind.SERVER);
			try (Scope scope = span.map(Span::makeCurrent).orElse(null)) {
				span.ifPresent(s -> {
					s.updateName("Outbox Collector (" + target + ")");
					s.setAttribute(OpenTelemetryUtils.CDS_TENANT, tenant);
					s.setAttribute(OpenTelemetryUtils.CDS_OUTBOX_TARGET, target);
				});

				if (withEmptyCheck) {
					boolean isEmpty = runtime.requestContext().featureToggles(STATIC_MODEL_ACCESS_FEATURE).systemUser(tenant).run(req -> {
						return outboxService.isEmpty();
					});
					if (isEmpty) {
						LOG.debug("The outbox '{}' for tenant '{}' is empty", target, tenant);
						return null;
					}
				}

				LOG.debug("Processing tenant '{}' in collector '{}'", tenant, target);
				TaskSchedule result = runtime.requestContext().systemUser(tenant).run(req -> {
					return runtime.changeSetContext().run(ctx -> {

						Predicate where = CQL.get(Messages.TARGET).eq(target)
								.and(CQL.get(Messages.ATTEMPTS).lt(CQL.constant(maxPublishAttempts)));

						long skip = calculateOffset(where);
						CqnSelect select = Select.from(Messages_.class).where(where)
								.orderBy(e -> e.timestamp().asc(), e -> e.ID().asc())
								.limit(1, skip)
								.lock(0);

						Messages message = db.run(select).first(Messages.class).orElse(null);
						if (message != null) {
							int attempt = message.getAttempts();
							switch(publish(message)) {
							case SUCCESS:
								db.run(Delete.from(Messages_.class).byId(message.getId()));
								telemetryData.recordOutgoingMessages(tenant, 1);
								return new TaskSchedule(new CollectorTask(tenant, false));
							case INVALID_VERSION:
								// outdated instance we should not plan any tasks for this tenant
								suspendedTenants.add(tenant);
								return null;
							case FAILED:
								// backoff => delay execution because of an actual error
								return new TaskSchedule(new CollectorTask(tenant, false), getPauseMillis(attempt), false);
							}
						}
						// no message found => no rescheduling
						return null;
					});
				});

				// record statistics after processing outbox
				// TODO calculating statistics after every row might be a little too expensive
				telemetryData.recordStatistics(runtime, db, tenant);
				return result;
			} catch (Exception e) {
				OpenTelemetryUtils.recordException(span, e);

				if (isLockTimeoutException(e)) {
					LOG.debug("Collector '{}' timed out waiting for table lock for tenant '{}'", target, tenant);
					if (!ordered) {
						// bad luck, retry immediately
						return new TaskSchedule(new CollectorTask(tenant, false));
					}
				} else if (TenantUtils.isUnknownTenant(e)) {
					LOG.debug("Unknown tenant '{}' for the outbox collector '{}'", tenant, target);
				} else {
					LOG.warn("Exception occurred for tenant '{}' in collector '{}'", tenant, target, e);
				}
				// unexpected technical error or unknown tenant
				// or another instance is already processing this ordered outbox
				return null;
			} finally {
				OpenTelemetryUtils.endSpan(span);
			}
		}
	}

	private long calculateOffset(Predicate whereClause) {
		if (this.ordered) {
			return 0;
		}

		CqnSelect select = Select
				.from(Messages_.class)
				.columns(CQL.count().as("count"))
				.where(whereClause);
		Result res = db.run(select);
		long count = ((Number) res.single().get("count")).longValue();
		// no secure random number needed, because the offset is not used for security relevant operations
		long offset = count < 2 ? 0 : ThreadLocalRandom.current().nextLong(count); // NOSONAR
		LOG.debug("Calculated offset for unordered processing of outbox collector '{}' is {}", target, offset);
		return offset;
	}

	@SuppressWarnings("unchecked")
	private PublishState publish(final Messages msg) {
		Map<String, Object> message = JsonParser.map(JsonParser.parseJson(msg.getMsg()));
		String outboxEvent = (String) message.get(PersistentOutbox.ATTR_EVENT);
		String messageVersion = (String) message.get(PersistentOutbox.ATTR_VERSION);
		message = (Map<String, Object>) message.get(PersistentOutbox.ATTR_MESSAGE);

		// check message version
		if (checkVersion && !Objects.equals(appVersion, messageVersion)) {
			Version messageVersionParsed = VersionUtil.parseVersion(messageVersion, null, null);
			if (appVersionParsed.compareTo(messageVersionParsed) < 0) {
				LOG.debug("Found newer version '{}' in outbox message. Suspending collector '{}' with version '{}'.", messageVersion, target, appVersion);
				return PublishState.INVALID_VERSION;
			}
		}

		LOG.debug("Publishing outbox message to outbox '{}' with event '{}'", target, outboxEvent);
		OutboxMessageEventContext ctx = OutboxMessageEventContext.create(outboxEvent);
		ctx.setIsInbound(true);
		ctx.setTimestamp(msg.getTimestamp());
		ctx.setMessage(Struct.access(message).as(OutboxMessage.class));

		try {
			outboxService.emit(ctx);
			return PublishState.SUCCESS;
		} catch (Throwable e) { // NOSONAR
			LOG.warn("Failed to emit outbox message with id '{}' to outbox '{}' with event '{}'", msg.getId(), target, outboxEvent, e);

			// record failure
			Map<String, Object> data = new HashMap<>();
			data.put(Messages.ATTEMPTS, msg.getAttempts() + 1);
			data.put(Messages.LAST_ATTEMPT_TIMESTAMP, Instant.now());

			if (storeLastError) {
				StringWriter stringWriter = new StringWriter();
				e.printStackTrace(new PrintWriter(stringWriter));
				data.put(Messages.LAST_ERROR, stringWriter.toString());
			}

			db.run(Update.entity(Messages_.class).data(data).byId(msg.getId()));
			return PublishState.FAILED;
		}
	}

	private long getPauseMillis(int attempt) {
		long retryInMillis = Math.round(Math.pow(2d, attempt) * 1000 + ThreadLocalRandom.current().nextLong(1001));
		return Math.min(retryInMillis, maxFailPause);
	}

	private static boolean isLockTimeoutException(Throwable t) {
		while (t != null) {
			if (t instanceof CdsLockTimeoutException) {
				return true;
			}
			t = t.getCause();
		}
		return false;
	}

	private static enum PublishState {
		SUCCESS,
		FAILED,
		INVALID_VERSION
	}

}
