/*
 * © 2021-2025 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.impl;

import com.sap.cds.ql.StructuredType;
import com.sap.cds.ql.cqn.CqnReference;
import com.sap.cds.ql.cqn.Path;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.services.CoreFactory;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.Service;
import com.sap.cds.services.ServiceException;
import com.sap.cds.services.ServiceExceptionUtils;
import com.sap.cds.services.changeset.ChangeSetContextAccessor;
import com.sap.cds.services.changeset.ChangeSetContextSPI;
import com.sap.cds.services.environment.PropertiesProvider;
import com.sap.cds.services.impl.changeset.ChangeSetContextAccessorImpl;
import com.sap.cds.services.impl.changeset.ChangeSetContextImpl;
import com.sap.cds.services.impl.messages.MessageImpl;
import com.sap.cds.services.impl.messages.MessageLookupImpl;
import com.sap.cds.services.impl.messages.MessageTargetImpl;
import com.sap.cds.services.impl.outbox.OutboxedServiceProxyUtils;
import com.sap.cds.services.impl.request.FeatureTogglesInfoFactory;
import com.sap.cds.services.impl.request.ParameterInfoFactory;
import com.sap.cds.services.impl.request.RequestContextAccessorImpl;
import com.sap.cds.services.impl.request.UserInfoFactory;
import com.sap.cds.services.impl.runtime.CdsRuntimeConfigurerImpl;
import com.sap.cds.services.messages.Message;
import com.sap.cds.services.messages.Message.Severity;
import com.sap.cds.services.messages.MessageLookup;
import com.sap.cds.services.messages.MessageTarget;
import com.sap.cds.services.request.FeatureToggle;
import com.sap.cds.services.request.FeatureTogglesInfo;
import com.sap.cds.services.request.ModifiableParameterInfo;
import com.sap.cds.services.request.ModifiableUserInfo;
import com.sap.cds.services.request.ParameterInfo;
import com.sap.cds.services.request.RequestContextAccessor;
import com.sap.cds.services.request.UserInfo;
import com.sap.cds.services.runtime.CdsRuntimeConfigurer;
import java.util.List;
import java.util.Map;
import java.util.function.Function;

public class CoreFactoryImpl implements CoreFactory {

  @Override
  public EventContext createEventContext(String event, String entityName) {
    return new EventContextImpl(event, entityName);
  }

  @Override
  public Service createService(String name) {
    return new ServiceImpl(name);
  }

  @Override
  public Service createService(String name, Service delegator) {
    return new ServiceImpl(name, delegator);
  }

  @Override
  public CdsRuntimeConfigurer createCdsRuntimeConfigurer(PropertiesProvider propertiesProvider) {
    return new CdsRuntimeConfigurerImpl(propertiesProvider);
  }

  @Override
  public RequestContextAccessor createRequestContextAccessor() {
    return new RequestContextAccessorImpl();
  }

  @Override
  public ChangeSetContextAccessor createChangeSetContextAccessor() {
    return new ChangeSetContextAccessorImpl();
  }

  @Override
  public ChangeSetContextSPI createChangeSetContextSPI() {
    return ChangeSetContextImpl.open(true);
  }

  @Override
  public ModifiableParameterInfo createParameterInfo(ParameterInfo parameterInfo) {
    return ParameterInfoFactory.create(parameterInfo);
  }

  @Override
  public ModifiableUserInfo createUserInfo(UserInfo userInfo) {
    return UserInfoFactory.create(userInfo);
  }

  @Override
  public FeatureTogglesInfo createFeatureTogglesInfo(List<FeatureToggle> featureToggles) {
    return FeatureTogglesInfoFactory.create(featureToggles);
  }

  @Override
  public FeatureTogglesInfo createFeatureTogglesInfo(Map<String, Boolean> featureToggles) {
    return FeatureTogglesInfoFactory.create(featureToggles);
  }

  @Override
  public ServiceExceptionUtils createServiceExceptionUtils() {
    return new ServiceExceptionUtilsImpl();
  }

  @Override
  public Message createMessage(Severity severity, String text) {
    return new MessageImpl(severity, text);
  }

  @Override
  public Message createMessage(String text, ServiceException exception) {
    return new MessageImpl(Severity.ERROR, text)
        .target(exception.getMessageTarget())
        .additionalTargets(exception.getAdditionalTargets())
        .code(exception.getErrorStatus().getCodeString())
        .longTextUrl(exception.getLongTextUrl())
        .transition(exception.isTransition());
  }

  @Override
  public Message createMessage(Severity severity, String text, Message message) {
    return new MessageImpl(severity, text)
        .target(message.getTarget())
        .additionalTargets(message.getAdditionalTargets())
        .code(message.getCode())
        .longTextUrl(message.getLongTextUrl())
        .transition(message.isTransition());
  }

  @Override
  public MessageTarget createMessageTarget(String target) {
    return MessageTargetImpl.create(target);
  }

  @Override
  public MessageTarget createMessageTarget(
      String parameter, Function<StructuredType<?>, Object> path) {
    return MessageTargetImpl.create(parameter, path);
  }

  @Override
  public <E extends StructuredType<E>> MessageTarget createMessageTarget(
      String parameter, Class<E> type, Function<E, Object> path) {
    return MessageTargetImpl.create(parameter, type, path);
  }

  @Override
  public MessageTarget createMessageTarget(Path path, CdsElement element) {
    return MessageTargetImpl.create(path, element);
  }

  @Override
  public MessageTarget createMessageTarget(String parameter, CqnReference ref) {
    return MessageTargetImpl.create(parameter, ref);
  }

  @Override
  public MessageLookup createMessageLookup(String messageOrKey, Object[] args) {
    return new MessageLookupImpl(messageOrKey, args);
  }

  @Override
  public <S extends Service> S unboxed(S service) {
    return OutboxedServiceProxyUtils.unboxed(service);
  }
}
