/*
 * © 2021-2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.impl.auditlog;

import com.sap.cds.ql.CQL;
import com.sap.cds.ql.ElementRef;
import com.sap.cds.ql.StructuredType;
import com.sap.cds.ql.cqn.CqnSelectListItem;
import com.sap.cds.ql.cqn.CqnSelectListValue;
import com.sap.cds.reflect.CdsAssociationType;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsStructuredType;
import com.sap.cds.services.utils.model.CdsAnnotations;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * This class provides utility methods required for Personal Data analysis of an {@link
 * CdsStructuredType entity}.
 */
class PersonalDataUtils {

  static final String PERSONAL_DATA_PREFIX = "@audit:";

  private final CdsStructuredType entity;
  private final PersonalDataCaches caches;
  private Boolean personalDataDeleted;
  private Set<CqnSelectListValue> resolveDataSubjectId;
  private Function<?, CqnSelectListItem>[] expandCompositionsForDelete;

  PersonalDataUtils(CdsStructuredType entity, PersonalDataCaches caches) {
    this.entity = Objects.requireNonNull(entity, "entity must not be null");
    this.caches = Objects.requireNonNull(caches, "caches must not be null");
  }

  /**
   * Indicates whether this entity could contain some Personal Data if an instance is deleted.
   *
   * @return <code>true</code> if entity could contain Personal Data if an instance is deleted
   */
  boolean hasPersonalDataDeleted() {
    if (this.personalDataDeleted == null) {
      this.personalDataDeleted = hasPersonalDataDeleted(this.entity);
    }
    return this.personalDataDeleted;
  }

  /**
   * A set of {@link CqnSelectListValue} that contains all data subject IDs.
   *
   * @return a set of {@link CqnSelectListValue}
   */
  Set<CqnSelectListValue> resolveDataSubjectId() {
    if (this.resolveDataSubjectId == null) {
      this.resolveDataSubjectId =
          Collections.unmodifiableSet(resolveDataSubjectId(this.entity, null));
    }
    return this.resolveDataSubjectId;
  }

  /**
   * Expands compositions and associations with cascade delete which could contain personal data in
   * case of a deletion.
   *
   * @return an array of expands
   */
  @SuppressWarnings("unchecked")
  <T> Function<T, CqnSelectListItem>[] expandCompositionsForDelete() {
    if (this.expandCompositionsForDelete == null) {
      this.expandCompositionsForDelete = expandCompositionsForDelete(this.entity, new HashSet<>());
    }
    // return a copy to avoid modifications by consumer on original array
    return (Function<T, CqnSelectListItem>[])
        Arrays.copyOf(this.expandCompositionsForDelete, this.expandCompositionsForDelete.length);
  }

  private Set<CqnSelectListValue> resolveDataSubjectId(
      CdsStructuredType entity, StructuredType<?> path) {
    Set<CqnSelectListValue> result = new HashSet<>();
    Map<CdsElement, List<CdsElement>> associations = new HashMap<>();
    PersonalDataMeta pdMeta = getMeta(entity);

    // get all assocs with datasubject ID
    pdMeta.getDataSubjectIds().stream()
        .filter(element -> element.getType().isAssociation())
        .forEach(element -> associations.put(element, null));

    pdMeta.getDataSubjectIds().stream()
        .filter(element -> element.getType().isSimple())
        .forEach(
            element -> {
              String assoc = CdsAnnotations.ODATA_FOREIGN_KEY_FOR.getOrValue(element, null);
              if (assoc != null) {
                CdsElement assocElement = entity.getAssociation(assoc);
                if (associations.get(assocElement) != null) {
                  associations.get(assocElement).add(element);
                } else {
                  List<CdsElement> foreignKeys = new LinkedList<>();
                  foreignKeys.add(element);
                  associations.put(assocElement, foreignKeys);
                }
              } else {
                result.add(appendElement(path, element.getName(), element.getName()));
              }
            });

    associations.entrySet().stream()
        .forEach(
            entry -> {
              CdsElement assoc = entry.getKey();
              if (foreignKeysAreDataSubjectIds(entity, assoc, entry.getValue())) {
                entry
                    .getValue()
                    .forEach(
                        element ->
                            result.add(
                                appendElement(
                                    path, element.getName(), extractKeyName(element, assoc))));
              } else {
                String assocName = assoc.getName();
                StructuredType<?> newPath = path != null ? path.to(assocName) : CQL.to(assocName);
                result.addAll(resolveDataSubjectId(entity.getTargetOf(assocName), newPath));
              }
            });
    return result;
  }

  private String extractKeyName(CdsElement element, CdsElement assoc) {
    String name = element.getName();
    return name.replace(assoc.getName() + "_", "");
  }

  private CqnSelectListValue appendElement(StructuredType<?> path, String element, String idName) {
    ElementRef<?> ref = path != null ? path.get(element) : CQL.get(element);
    return ref.as(PersonalDataUtils.getDSIdAlias(idName));
  }

  private boolean foreignKeysAreDataSubjectIds(
      CdsStructuredType entity, CdsElement assoc, List<CdsElement> foreignKeys) {
    CdsEntity target = entity.getTargetOf(assoc.getName());
    PersonalDataMeta pdMeta = getMeta(target);
    if (!pdMeta.isDataSubject()) {
      return false;
    }
    if (foreignKeys == null || pdMeta.getDataSubjectIds().size() != foreignKeys.size()) {
      return false;
    }
    Set<String> fkNames = foreignKeys.stream().map(CdsElement::getName).collect(Collectors.toSet());
    return pdMeta.getDataSubjectIds().stream()
        .allMatch(key -> fkNames.contains(assoc.getName() + "_" + key.getName()));
  }

  /**
   * Indicates whether the given entity could contain personal data fields in case of a deletion.
   *
   * @param entity the {@link CdsStructuredType entity} whose personal data fields will be
   *     determined
   * @return <code>true</code> if personal data fields are found
   */
  private boolean hasPersonalDataDeleted(CdsStructuredType entity) {
    boolean hasPDdeleted = getMeta(entity).hasPersonalData();
    if (!hasPDdeleted) {
      hasPDdeleted =
          getAssociationsWithPersonalData(entity) //
              .map(element -> element.getType().as(CdsAssociationType.class)) //
              .anyMatch(assoc -> hasPersonalDataDeleted(assoc.getTarget()));
    }
    return hasPDdeleted;
  }

  @SuppressWarnings("unchecked")
  private <T> Function<T, CqnSelectListItem>[] expandCompositionsForDelete(
      CdsStructuredType entity, Set<String> expanded) {
    if (expanded.contains(entity.getQualifiedName())) {
      return Collections.emptyList().toArray(new Function[0]);
    } else {
      expanded.add(entity.getQualifiedName());
    }

    List<Function<StructuredType<?>, CqnSelectListItem>> columns = new ArrayList<>();
    columns.add(StructuredType::_all);
    getAssociationsWithPersonalData(entity)
        .forEach(
            element -> {
              CdsEntity targetOf = entity.getTargetOf(element.getName());
              columns.add(
                  c ->
                      c.to(element.getName())
                          .expand(expandCompositionsForDelete(targetOf, expanded)));
            });
    return columns.toArray(new Function[columns.size()]);
  }

  /**
   * Returns an element stream of compositions or associations with cascade delete containing
   * personal data.
   *
   * @param entity the entity to analyse
   * @return a {@link Stream} of {@link CdsElement}
   */
  private Stream<CdsElement> getAssociationsWithPersonalData(CdsStructuredType entity) {
    return entity
        .associations()
        // filter: only compositions or association (with cascade delete) with personal data
        // annotations
        .filter(
            element -> {
              CdsAssociationType assocType = element.getType().as(CdsAssociationType.class);
              return CdsAnnotations.CASCADE_DELETE.getOrValue(element, assocType.isComposition())
                  && getMeta(assocType.getTarget()).hasPersonalData();
            });
  }

  // cache access

  private PersonalDataMeta getMeta(CdsStructuredType entity) {
    return this.caches.getMeta(entity);
  }

  // static helpers

  static String getDSIdAlias(String id) {
    return PERSONAL_DATA_PREFIX + "DS_" + id;
  }

  static String getKeyAlias(String key) {
    return PERSONAL_DATA_PREFIX + key;
  }
}
