/*
 * © 2025 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.impl.cds;

import static com.sap.cds.services.utils.model.CdsAnnotations.NON_EXPANDABLE_PROPERTIES;

import com.sap.cds.ql.cqn.CqnExpand;
import com.sap.cds.ql.cqn.CqnSelectListItem;
import com.sap.cds.reflect.CdsStructuredType;
import com.sap.cds.services.cds.ApplicationService;
import com.sap.cds.services.cds.CdsReadEventContext;
import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.Before;
import com.sap.cds.services.handler.annotations.HandlerOrder;
import com.sap.cds.services.handler.annotations.ServiceName;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.OrderConstants;
import com.sap.cds.services.utils.model.CdsAnnotations;
import com.sap.cds.util.CdsModelUtils;
import java.util.Collection;

@ServiceName(value = "*", type = ApplicationService.class)
public class ExpandRestrictionsHandler implements EventHandler {

  private final int defaultMaxLevelsLimit;

  public ExpandRestrictionsHandler(CdsRuntime runtime) {
    this.defaultMaxLevelsLimit =
        runtime
            .getEnvironment()
            .getCdsProperties()
            .getQuery()
            .getRestrictions()
            .getExpand()
            .getMaxLevels();
  }

  @Before
  @HandlerOrder(OrderConstants.Before.CHECK_STATEMENT)
  void enforceExpandRestrictions(CdsReadEventContext context) {
    ExpandValidator checker = new ExpandValidator(context.getTarget(), -1);
    context.getCqn().items().stream()
        .filter(CqnSelectListItem::isExpand)
        .forEach(i -> checker.validate(i.asExpand()));
  }

  private class ExpandValidator {
    private final CdsStructuredType entity;
    private final Collection<String> nonExpandableProperties;

    // limit -1 means absence of limit and must be carried over to child expands
    private final int limit;

    private ExpandValidator(CdsStructuredType entity, int remainingLimit) {
      this.entity = entity;
      this.nonExpandableProperties = NON_EXPANDABLE_PROPERTIES.asCollectionOfValues(entity);

      int currentLimit = CdsAnnotations.EXPAND_MAX_LEVELS.getOrValue(entity, defaultMaxLevelsLimit);
      if (remainingLimit < 0 || (currentLimit >= 0 && currentLimit < remainingLimit)) {
        limit = currentLimit;
      } else {
        limit = remainingLimit;
      }
    }

    public void validate(CqnExpand expand) {
      if (!CdsAnnotations.EXPANDABLE.isTrue(entity)) {
        throw new ErrorStatusException(CdsErrorStatuses.NOT_EXPANDABLE, entity.getQualifiedName());
      }

      if (nonExpandableProperties.contains(expand.ref().firstSegment())) {
        throw new ErrorStatusException(
            CdsErrorStatuses.NON_EXPANDABLE_PROPERTY,
            expand.ref().firstSegment(),
            entity.getQualifiedName());
      }

      if (limit == 0) {
        throw new ErrorStatusException(CdsErrorStatuses.TOO_MANY_EXPANDS);
      }
      int nextLimit = limit > 0 ? limit - 1 : limit;

      CdsStructuredType target = CdsModelUtils.getTargetOf(entity, expand.ref().path());
      ExpandValidator checker = new ExpandValidator(target, nextLimit);
      expand.items().stream()
          .filter(CqnSelectListItem::isExpand)
          .forEach(i -> checker.validate(i.asExpand()));
    }
  }
}
