/*
 * © 2019-2025 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.impl.cds;

import static com.sap.cds.ql.impl.SelectListValueBuilder.select;

import com.sap.cds.Result;
import com.sap.cds.impl.DataProcessor;
import com.sap.cds.ql.CQL;
import com.sap.cds.ql.Select;
import com.sap.cds.ql.Selectable;
import com.sap.cds.ql.cqn.CqnSelect;
import com.sap.cds.ql.cqn.CqnSelectListValue;
import com.sap.cds.ql.cqn.CqnStatement;
import com.sap.cds.reflect.CdsAssociationType;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsModel;
import com.sap.cds.reflect.CdsStructuredType;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.cds.ApplicationService;
import com.sap.cds.services.cds.CqnService;
import com.sap.cds.services.draft.DraftService;
import com.sap.cds.services.draft.Drafts;
import com.sap.cds.services.environment.CdsProperties;
import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.Before;
import com.sap.cds.services.handler.annotations.HandlerOrder;
import com.sap.cds.services.handler.annotations.ServiceName;
import com.sap.cds.services.impl.utils.CdsServiceUtils;
import com.sap.cds.services.persistence.PersistenceService;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.OrderConstants;
import com.sap.cds.services.utils.TenantAwareCache;
import com.sap.cds.services.utils.model.CdsAnnotations;
import com.sap.cds.util.CdsModelUtils;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;

@ServiceName(value = "*", type = ApplicationService.class)
public class ReadOnlyHandler implements EventHandler {

  private static final String READONLY_PREFIX = "@readonly_";

  private final CdsProperties cdsProperties;
  private final PersistenceService db;
  private final TenantAwareCache<Map<CdsEntity, List<CqnSelectListValue>>, CdsModel>
      annotationCache;

  public ReadOnlyHandler(CdsRuntime runtime) {
    this.cdsProperties = runtime.getEnvironment().getCdsProperties();
    this.db =
        runtime
            .getServiceCatalog()
            .getService(PersistenceService.class, PersistenceService.DEFAULT_NAME);
    this.annotationCache = TenantAwareCache.create(ConcurrentHashMap::new, runtime);
  }

  @Before(
      event = {
        CqnService.EVENT_CREATE,
        CqnService.EVENT_UPDATE,
        CqnService.EVENT_UPSERT,
        DraftService.EVENT_DRAFT_NEW,
        DraftService.EVENT_DRAFT_PATCH
      })
  @HandlerOrder(OrderConstants.Before.FILTER_FIELDS)
  public void cleanReadOnlyFields(EventContext context, CqnStatement cqn) {
    boolean enforceReadonlyOnDrafts =
        context.getCdsRuntime().getEnvironment().getCdsProperties().getDrafts().isEnforceReadonly();
    if (!enforceReadonlyOnDrafts && context.getEvent().startsWith("DRAFT_")) {
      return; // don't evaluate readonly on draft instances (DRAFT_NEW resp. DRAFT_PATCH)
    }

    boolean isDraftActivate =
        enforceReadonlyOnDrafts
            && Boolean.TRUE.equals(cqn.hints().get(DraftService.EVENT_DRAFT_SAVE));
    Map<CdsEntity, List<CqnSelectListValue>> annotationsPerEntity = annotationCache.findOrCreate();
    Map<CdsEntity, List<Map<String, Object>>> keysPerEntity = new HashMap<>();
    List<Map<String, Object>> dataList = CdsServiceUtils.getEntities(context);

    // TODO handle structured types
    DataProcessor.create()
        .bulkAction(
            (type, entries) -> {
              // collect all dynamic readonly annotations
              // dynamic readonly is not evaluated on CREATE and DRAFT_NEW
              // In case of draftActivate dynamic readonly has been evaluated on the draft already
              if (type instanceof CdsEntity entity && !cqn.isInsert() && !isDraftActivate) {
                List<CqnSelectListValue> readonlyExpressions =
                    annotationsPerEntity.computeIfAbsent(entity, this::collectReadonlyExpressions);

                if (!readonlyExpressions.isEmpty()) {
                  List<Map<String, Object>> keyValues =
                      keysPerEntity.computeIfAbsent(entity, t -> new ArrayList<>());
                  Set<String> keyNames = CdsModelUtils.concreteKeyNames(type);
                  entries.forEach(
                      entry -> ConstraintAssertionHandler.addKeysTo(keyNames, entry, keyValues));
                }
              }

              // remove static readonly elements directly from the map
              for (Map<String, Object> entry : entries) {
                type.elements()
                    .filter(
                        element ->
                            isStaticallyReadOnly(
                                element, context.getEvent(), isDraftActivate, entry))
                    .forEach(e -> entry.remove(e.getName()));
              }
            })
        .process(dataList, context.getTarget());

    // evaluate readonly expressions
    Map<CdsStructuredType, Map<Map<String, Object>, List<String>>> dynamicReadonlyPerEntity =
        new HashMap<>(keysPerEntity.size());
    keysPerEntity.forEach(
        (entity, keyValues) -> {
          if (keyValues.isEmpty()) return;

          // put readonly conditions on select list
          // keys need to be part of the select list to map readonly results to entity instances
          List<Selectable> selectables = new ArrayList<>(annotationsPerEntity.get(entity));
          List<String> keyElements = new ArrayList<>(keyValues.get(0).keySet());
          keyElements.forEach(keyElement -> selectables.add(CQL.get(keyElement)));

          // execute a single query, which performs all checks at once on the DB using an `IN`
          // operator for the given entity
          String entityName =
              context.getEvent().startsWith("DRAFT_")
                  ? entity.getQualifiedName() + "_drafts"
                  : entity.getQualifiedName();
          CqnSelect query =
              Select.from(entityName).columns(selectables).where(CQL.in(keyElements, keyValues));
          Result readonlyResult = db.run(query);

          readonlyResult.forEach(
              row -> {
                Map<String, Object> keys = new HashMap<>(keyElements.size());
                List<String> dynamicReadonly =
                    new ArrayList<>(selectables.size() - keyElements.size());

                // go through the result to collect keys and readonly elements
                row.forEach(
                    (k, v) -> {
                      if (k.startsWith(READONLY_PREFIX) && Boolean.TRUE.equals(v)) {
                        dynamicReadonly.add(k.substring(READONLY_PREFIX.length()));
                      } else if (keyElements.contains(k)) {
                        keys.put(k, v);
                      }
                    });

                if (!dynamicReadonly.isEmpty()) {
                  dynamicReadonlyPerEntity
                      .computeIfAbsent(entity, e -> new HashMap<>(keyValues.size()))
                      .put(keys, dynamicReadonly);
                }
              });
        });

    // remove dynamic readonly elements
    if (dynamicReadonlyPerEntity.isEmpty()) return;
    DataProcessor.create()
        .bulkAction(
            (type, entries) -> {
              Map<Map<String, Object>, List<String>> dynamicReadonly =
                  dynamicReadonlyPerEntity.get(type);
              if (dynamicReadonly == null) return;

              Set<String> keyNames = CdsModelUtils.concreteKeyNames(type);
              entries.forEach(
                  entry -> {
                    Map<String, Object> keyValues =
                        keyNames.stream().collect(Collectors.toMap(keyName -> keyName, entry::get));
                    List<String> readonlyElements = dynamicReadonly.get(keyValues);
                    if (readonlyElements != null) {
                      readonlyElements.forEach(entry::remove);
                    }
                  });
            })
        .process(dataList, context.getTarget());
  }

  private List<CqnSelectListValue> collectReadonlyExpressions(CdsStructuredType type) {
    List<CqnSelectListValue> readonlyExpressions = new ArrayList<>();
    type.elements()
        .forEach(
            e -> {
              if (CdsAnnotations.READONLY.isExpression(e)) {
                // elements with readonly expressions that evaluate to UNKNOWN (due to null) are
                // considered writable
                readonlyExpressions.add(
                    select(CdsAnnotations.READONLY.asPredicate(e))
                        .as(READONLY_PREFIX + e.getName())
                        .build());
              }
            });
    return readonlyExpressions;
  }

  /**
   * Returns true, if a {@link CdsElement} is read only
   *
   * @param element the {@link CdsElement}
   * @param event the event currently being processed
   * @param isDraftActivate true, if the event is triggered by draftActivate
   * @param map the data map
   * @return true, if a {@link CdsElement} is read only
   */
  public boolean isStaticallyReadOnly(
      CdsElement element, String event, boolean isDraftActivate, Map<String, Object> map) {
    // cds.on.insert/update is always readonly and also cleaned on draftActivate
    // this ensures that managed values are properly updated during draftActivate
    if (CdsAnnotations.ON_UPDATE.getOrDefault(element) != null
        || CdsAnnotations.ON_INSERT.getOrDefault(element) != null) {
      return true;
    }

    if (CdsAnnotations.READONLY.isExpression(element)) {
      // handled by dynamic @readonly for UPDATE-like events
      return false;
    }

    // Check @readonly like annotations
    // In case of draftActivate we don't remove readonly values again, as they might have been
    // calculated on server-side
    if (!isDraftActivate
        && (isElementAnnotatedWithStaticReadonly(element)
            || CdsAnnotations.FIELD_CONTROL_READONLY.isTrue(element)
            || (CdsAnnotations.COMMON_FIELDCONTROL.getOrDefault(element)
                    instanceof Map<?, ?> commonFieldControl
                && "ReadOnly".equals(commonFieldControl.get("#"))))) {
      return true;
    }

    // keys are immutable in CDS4j and computed if type = UUID or via onInsert annotation
    if (!element.isKey()) {
      // Core.Computed is read only
      // onUpdate and onInsert are also Core.Computed, but already handled above
      // In case of draftActivate we don't remove computed values again
      if (!isDraftActivate && CdsAnnotations.CORE_COMPUTED.isTrue(element)) {
        return true;
      }

      // Core.Immutable is read only during update
      // TODO For UPSERT that is turned into an UPDATE immutable should be considered
      // TODO For a deep UPDATE that is turned into an INSERT immutable should NOT be considered
      // For a deep UPDATE on draft-enabled entities the condition HAS_ACTIVE_ENTITY = false
      // is already used to know if something in the deep update is actually an insert
      // however that is not safe, as anybody could set that HAS_ACTIVE_ENTITY = false from the
      // outside
      // For draftActivate we also enforce immutable, as DRAFT_PATCH always accepts also immutable
      // values
      // this ensures that we can edit immutable fields until the first draft activation
      boolean coreImmutable = CdsAnnotations.CORE_IMMUTABLE.isTrue(element);
      return coreImmutable
          && CqnService.EVENT_UPDATE.equals(event)
          && !Boolean.FALSE.equals(map.get(Drafts.HAS_ACTIVE_ENTITY));
    }
    return false;
  }

  private boolean isElementAnnotatedWithStaticReadonly(CdsElement element) {
    if (CdsAnnotations.READONLY.isTrue(element)) {
      return true;
    }
    if (element.getType().isAssociation()
        && cdsProperties.getQuery().getDeepEntityReadonly().isEnabled()) {
      CdsEntity targetType = element.getType().as(CdsAssociationType.class).getTarget();
      return CdsAnnotations.READONLY.isTrue(targetType);
    }
    return false;
  }
}
