/*
 * © 2022-2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.impl.cds;

import static com.sap.cds.impl.DataProcessor.create;
import static com.sap.cds.ql.cqn.CqnLock.Mode.SHARED;
import static com.sap.cds.services.impl.utils.ValidatorErrorUtils.handleValidationError;
import static com.sap.cds.services.utils.model.CdsAnnotations.ASSERT_TARGET;
import static com.sap.cds.util.CdsModelUtils.CascadeType.INSERT;
import static com.sap.cds.util.CdsModelUtils.CascadeType.UPDATE;
import static com.sap.cds.util.CdsModelUtils.isCascading;
import static com.sap.cds.util.CdsModelUtils.managedToOne;
import static java.util.Collections.emptyMap;
import static java.util.Objects.requireNonNull;

import com.sap.cds.impl.DataProcessor;
import com.sap.cds.ql.CQL;
import com.sap.cds.ql.cqn.CqnSelect;
import com.sap.cds.ql.cqn.CqnValue;
import com.sap.cds.ql.cqn.Path;
import com.sap.cds.ql.impl.PathImpl;
import com.sap.cds.ql.impl.SelectBuilder;
import com.sap.cds.reflect.CdsAssociationType;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsModel;
import com.sap.cds.reflect.CdsStructuredType;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.cds.ApplicationService;
import com.sap.cds.services.cds.CqnService;
import com.sap.cds.services.draft.Drafts;
import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.Before;
import com.sap.cds.services.handler.annotations.HandlerOrder;
import com.sap.cds.services.handler.annotations.ServiceName;
import com.sap.cds.services.impl.utils.CdsModelUtils;
import com.sap.cds.services.impl.utils.CdsServiceUtils;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.DraftUtils;
import com.sap.cds.services.utils.OrderConstants;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Predicate;

@ServiceName(value = "*", type = ApplicationService.class)
public class TargetAssertionHandler implements EventHandler {

  @Before(event = {CqnService.EVENT_CREATE, CqnService.EVENT_UPDATE, CqnService.EVENT_UPSERT})
  @HandlerOrder(OrderConstants.Before.VALIDATE_FIELDS)
  public void run(EventContext context) {
    List<Validation> validations = new LinkedList<>();
    create()
        .action(new Action(context.getModel(), validations))
        .process(CdsServiceUtils.getEntities(context), context.getTarget());

    // To avoid interference of messages, we first emit the check events and then add messages.
    validations.removeIf(
        check ->
            isTargetPresent((CqnService) context.getService(), check.getTarget(), check.getKeys()));
    validations.forEach(
        error -> {
          // In case of foreign key fields, the messages will be created per each field of the
          // original entry
          if (!error.getForeignKeyValues().isEmpty()) {
            error
                .getForeignKeyValues()
                .forEach(
                    item ->
                        handleValidationError(
                            context,
                            error.getPath(),
                            error.getSource().getElement(item),
                            null,
                            CdsErrorStatuses.TARGET_ENTITY_MISSING));
          } else {
            handleValidationError(
                context,
                error.getPath(),
                error.getAssociation(),
                null,
                CdsErrorStatuses.TARGET_ENTITY_MISSING);
          }
        });
  }

  private static class Action implements DataProcessor.Action {
    private final CdsModel model;
    private final List<Validation> validations;

    private static final Predicate<CdsElement> isAssociationRelevant =
        assoc ->
            requireNonNull(ASSERT_TARGET.getOrValue(assoc, false))
                && managedToOne(assoc.getType())
                && !isCascading(INSERT, assoc)
                && !isCascading(UPDATE, assoc);

    public Action(CdsModel model, List<Validation> validations) {
      this.model = model;
      this.validations = validations;
    }

    @Override
    public void entry(
        Path path, CdsElement element, CdsStructuredType type, Map<String, Object> sourceData) {
      type.associations()
          .filter(isAssociationRelevant)
          .forEach(
              assoc -> {
                String assocName = assoc.getName();
                @SuppressWarnings("unchecked")
                Map<String, Object> targetData =
                    (Map<String, Object>) sourceData.getOrDefault(assocName, emptyMap());
                Map<String, Object> targetPKs = new HashMap<>();
                Set<String> fkNames = new HashSet<>();

                Map<String, CqnValue> fkMapping =
                    CdsModelUtils.getFkMapping(this.model, assoc, false);
                fkMapping.forEach(
                    (fk, pkRef) -> {
                      String pk = pkRef.asRef().lastSegment();

                      Object fkValue = sourceData.get(fk);
                      if (fkValue != null) {
                        targetPKs.put(pk, fkValue);
                        fkNames.add(fk);
                      }

                      // pk from target shadows fk
                      Object pkValue = targetData.get(pk);
                      if (pkValue != null) {
                        targetPKs.put(pk, pkValue);
                      }
                    });

                // Only accept entry for the check if all keys are present in the content
                if (targetPKs.size() == fkMapping.size()) {
                  validations.add(
                      new Validation(
                          ((PathImpl) path).append(element, type, sourceData),
                          assoc,
                          targetPKs,
                          fkNames));
                }
              });
    }
  }

  private boolean isTargetPresent(
      CqnService service, CdsEntity target, Map<String, Object> content) {
    if (DraftUtils.isDraftEnabled(target)) {
      content.put(Drafts.IS_ACTIVE_ENTITY, true);
    }

    CqnSelect existenceCheck =
        SelectBuilder.from(target.getQualifiedName())
            .hint(SelectBuilder.IGNORE_LOCALIZED_VIEWS, true)
            .columns(CQL.constant(1).as("ID"))
            .limit(1)
            .matching(content)
            .lock(SHARED);
    return service.run(existenceCheck).first().isPresent();
  }

  private static class Validation {
    private final Path path;
    private final CdsElement association;
    private final Map<String, Object> keys;
    private final Set<String> foreignKeyNames;

    private Validation(
        Path path, CdsElement association, Map<String, Object> keys, Set<String> foreignKeyNames) {
      this.path = path;
      this.association = association;
      this.keys = keys;
      this.foreignKeyNames = foreignKeyNames;
    }

    public Set<String> getForeignKeyValues() {
      return foreignKeyNames;
    }

    public CdsStructuredType getSource() {
      return association.getDeclaringType();
    }

    public CdsEntity getTarget() {
      return association.getType().as(CdsAssociationType.class).getTarget();
    }

    public Path getPath() {
      return path;
    }

    public CdsElement getAssociation() {
      return association;
    }

    public Map<String, Object> getKeys() {
      return keys;
    }
  }
}
