/*
 * © 2020-2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.impl.draft;

import static com.sap.cds.services.impl.model.DynamicModelProvider.STATIC_MODEL_ACCESS_FEATURE;

import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.sap.cds.CdsCommunicationException;
import com.sap.cds.Result;
import com.sap.cds.ResultBuilder;
import com.sap.cds.ql.CQL;
import com.sap.cds.ql.Delete;
import com.sap.cds.ql.Select;
import com.sap.cds.ql.cqn.CqnDelete;
import com.sap.cds.services.ServiceCatalog;
import com.sap.cds.services.application.ApplicationLifecycleService;
import com.sap.cds.services.application.ApplicationPreparedEventContext;
import com.sap.cds.services.application.ApplicationStoppedEventContext;
import com.sap.cds.services.cds.CqnService;
import com.sap.cds.services.draft.DraftAdministrativeData;
import com.sap.cds.services.draft.DraftGcEventContext;
import com.sap.cds.services.draft.DraftSaveEventContext;
import com.sap.cds.services.draft.DraftService;
import com.sap.cds.services.draft.Drafts;
import com.sap.cds.services.environment.CdsProperties;
import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.After;
import com.sap.cds.services.handler.annotations.HandlerOrder;
import com.sap.cds.services.handler.annotations.On;
import com.sap.cds.services.handler.annotations.ServiceName;
import com.sap.cds.services.mt.TenantProviderService;
import com.sap.cds.services.persistence.PersistenceService;
import com.sap.cds.services.request.RequestContext;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.DraftUtils;
import com.sap.cds.services.utils.OpenTelemetryUtils;
import com.sap.cds.services.utils.OrderConstants;
import com.sap.cds.services.utils.TenantAwareCache;
import io.opentelemetry.api.trace.Span;
import io.opentelemetry.api.trace.SpanKind;
import io.opentelemetry.context.Scope;
import java.time.Duration;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@ServiceName(value = ApplicationLifecycleService.DEFAULT_NAME)
public class DraftGCHandler implements EventHandler {

  private static final Logger LOG = LoggerFactory.getLogger(DraftGCHandler.class);
  private static final int THREAD_POOL_SIZE = 1;

  private final ExecutorService gcExecutor;
  private final TenantAwareCache<Map<String, Instant>, Boolean> lastGcs;
  private final CdsRuntime runtime;

  private Timer timer;

  public DraftGCHandler(CdsRuntime runtime) {
    this.runtime = runtime;
    this.gcExecutor =
        Executors.newFixedThreadPool(
            THREAD_POOL_SIZE,
            new ThreadFactoryBuilder().setNameFormat("draft-gc-exec-%d").setDaemon(true).build());
    this.lastGcs =
        TenantAwareCache.create(
            () -> RequestContext.getCurrent(runtime).getUserInfo().getTenant(),
            ConcurrentHashMap::new,
            () -> Boolean.TRUE);
  }

  @On
  protected void initializeGC(ApplicationPreparedEventContext context) {
    if (runtime.getEnvironment().getCdsProperties().getEnvironment().getCommand().isEnabled()) {
      return;
    }
    CdsProperties.Drafts.GC gcConfig =
        runtime.getEnvironment().getCdsProperties().getDrafts().getGc();
    if (gcConfig.isEnabled() && timer == null) {
      timer = new Timer("Draft GC Timer", true);
      // distribute the first execution time randomly to reduce the database load when
      // e.g. restarting the app
      long gcInterval = gcConfig.getInterval().toMillis();
      long firstExecutionTime = ThreadLocalRandom.current().nextLong(gcInterval);
      timer.schedule(
          new TimerTask() {
            @Override
            public void run() {
              try {
                gcAll();
              } catch (Throwable t) {
                LOG.error("Failed to gc drafts", t);
              }
            }
          },
          firstExecutionTime,
          gcInterval);
    }
  }

  @On
  protected void stopGC(ApplicationStoppedEventContext context) {
    if (timer != null) {
      timer.cancel();
      timer = null;
    }
  }

  @After(service = "*", serviceType = DraftService.class)
  protected void afterDraftSave(DraftSaveEventContext context) {
    CdsProperties.Drafts.GC gcConfig =
        context.getCdsRuntime().getEnvironment().getCdsProperties().getDrafts().getGc();
    if (gcConfig.isEnabled()) {
      String tenant = context.getUserInfo().getTenant();
      gcExecutor.execute(
          () -> {
            Instant now = Instant.now();

            DraftService draftService = context.getService();
            if (readyForGc(now, draftService, gcConfig.getInterval())) {
              try {
                context
                    .getCdsRuntime()
                    .requestContext()
                    .systemUser(tenant)
                    .privilegedUser()
                    .run(
                        reqContext -> {
                          lastGcs.findOrCreate().put(draftService.getName(), now);
                          draftService.gcDrafts();
                        });
              } catch (Exception e) {
                LOG.error(
                    "An error occurred while executing draft gc for tenant '{}' and service '{}' after draft save",
                    tenant,
                    draftService.getName(),
                    e);
              }
            }
          });
    }
  }

  @On(service = "*", serviceType = DraftService.class)
  @HandlerOrder(OrderConstants.On.DEFAULT_ON)
  protected void onGcDrafts(DraftGcEventContext context) {
    Optional<Span> span =
        OpenTelemetryUtils.createSpan(OpenTelemetryUtils.CdsSpanType.DRAFT_GC, SpanKind.SERVER);
    try (Scope scope = span.map(Span::makeCurrent).orElse(null)) {
      span.ifPresent(
          s -> {
            s.updateName("Draft GC (" + context.getService().getName() + ")");
            s.setAttribute(OpenTelemetryUtils.CDS_TENANT, context.getUserInfo().getTenant());
            s.setAttribute(OpenTelemetryUtils.CDS_SERVICE, context.getService().getName());
          });

      Instant threshold =
          getAgeThreshold(context.getCdsRuntime().getEnvironment().getCdsProperties());

      DraftService draftService = context.getService();
      AtomicLong numCancelledDrafts = new AtomicLong(0);

      context
          .getCdsRuntime()
          .requestContext()
          .privilegedUser()
          .run(
              ctx -> {
                draftService
                    .getDefinition()
                    .entities()
                    .forEach(
                        e -> {
                          if (DraftUtils.isDraftRoot(e)
                              && !e.getQualifiedName().endsWith(DraftModifier.DRAFT_SUFFIX)) {
                            CqnDelete deleteOldDrafts =
                                Delete.from(e)
                                    .where(
                                        c ->
                                            c.get(Drafts.IS_ACTIVE_ENTITY)
                                                .eq(false)
                                                .and(
                                                    c.to(Drafts.DRAFT_ADMINISTRATIVE_DATA)
                                                        .anyMatch(
                                                            a ->
                                                                a.get(
                                                                        DraftAdministrativeData
                                                                            .LAST_CHANGE_DATE_TIME)
                                                                    .le(threshold))));
                            Result result = draftService.cancelDraft(deleteOldDrafts);
                            if (result.rowCount() > 0) {
                              LOG.debug(
                                  "Draft GC deleted {} drafts of entity '{}'",
                                  result.rowCount(),
                                  e.getQualifiedName());
                              numCancelledDrafts.addAndGet(result.rowCount());
                            }
                          }
                        });
              });
      long cancelledDrafts = numCancelledDrafts.get();
      if (cancelledDrafts > 0) {
        LOG.info(
            "Draft GC deleted {} drafts of service '{}'", cancelledDrafts, draftService.getName());
      }
      context.setResult(ResultBuilder.deletedRows(cancelledDrafts).result());
    } catch (Exception e) {
      OpenTelemetryUtils.recordException(span, e);
      throw e;
    } finally {
      OpenTelemetryUtils.endSpan(span);
    }
  }

  protected void gcAll() {
    ServiceCatalog serviceCatalog = runtime.getServiceCatalog();
    CdsProperties.Drafts.GC gc = runtime.getEnvironment().getCdsProperties().getDrafts().getGc();
    Duration maxPause = gc.getMaxPause();
    Duration gcInterval = gc.getInterval();

    Instant ageThreshold = getAgeThreshold(runtime.getEnvironment().getCdsProperties());
    List<String> tenants =
        runtime
            .requestContext()
            .systemUserProvider()
            .run(
                r -> {
                  TenantProviderService tenantProvider =
                      r.getServiceCatalog()
                          .getService(
                              TenantProviderService.class, TenantProviderService.DEFAULT_NAME);
                  return tenantProvider.readTenants();
                });

    Collections.shuffle(tenants);
    List<DraftService> draftServices =
        serviceCatalog
            .getServices(DraftService.class)
            .collect(Collectors.toCollection(ArrayList::new));
    Collections.shuffle(draftServices);

    tenants.forEach(
        tenant -> {
          try {
            List<DraftService> servicesToGc = new ArrayList<>();
            draftServices.forEach(
                draftService -> {
                  Instant now = Instant.now();
                  if (readyForGc(now, draftService, gcInterval)) {
                    lastGcs.findOrCreate().put(draftService.getName(), now);
                    servicesToGc.add(draftService);
                  }
                });
            if (!servicesToGc.isEmpty()) {
              if (hasAnyDraftsOlderThanThreshold(runtime, tenant, ageThreshold)) {
                LOG.debug("Executing draft GC for tenant {}", tenant);
                runtime
                    .requestContext()
                    .systemUser(tenant)
                    .privilegedUser()
                    .run(
                        ctx -> {
                          servicesToGc.forEach(DraftService::gcDrafts);
                        });
              }
              TimeUnit.MILLISECONDS.sleep(
                  ThreadLocalRandom.current().nextLong(maxPause.toMillis()));
            }
          } catch (InterruptedException ie) {
            LOG.debug(
                "Draft GC timer thread '{}' interrupted", Thread.currentThread().getName(), ie);
            Thread.currentThread().interrupt();
          } catch (Exception e) {
            Throwable rootCause = ExceptionUtils.getRootCause(e);
            if (rootCause instanceof CdsCommunicationException cdsCommunicationException
                && cdsCommunicationException.getHttpStatusCode() == 404) {
              LOG.debug("Skipped draft GC for nonexistent tenant '{}'", tenant);
            } else {
              LOG.error("Failed to gc drafts of tenant '{}'", tenant, e);
            }
          }
        });
  }

  private boolean readyForGc(Instant now, DraftService draftService, Duration gcInterval) {
    Instant lastGc = lastGcs.findOrCreate().get(draftService.getName());
    return lastGc == null || lastGc.plus(gcInterval).isBefore(now);
  }

  private static Instant getAgeThreshold(CdsProperties configuration) {
    return Instant.now()
        .minus(configuration.getDrafts().getDeletionTimeout())
        .truncatedTo(ChronoUnit.MILLIS);
  }

  private static boolean hasAnyDraftsOlderThanThreshold(
      CdsRuntime runtime, String tenant, Instant ageThreshold) {
    return runtime
        .requestContext()
        // Use static model access to avoid fetching the model for the tenant
        .featureToggles(STATIC_MODEL_ACCESS_FEATURE)
        .systemUser(tenant)
        .run(
            requestContext -> {
              try {
                CqnService persistenceService =
                    (CqnService)
                        requestContext
                            .getServiceCatalog()
                            .getService(PersistenceService.DEFAULT_NAME);
                Result result =
                    persistenceService.run(
                        Select.from(DraftAdministrativeData.CDS_NAME)
                            .columns(CQL.constant("1").as("ID"))
                            .limit(1)
                            .where(
                                CQL.get(DraftAdministrativeData.LAST_CHANGE_DATE_TIME)
                                    .le(ageThreshold)));
                return result.rowCount() > 0;
              } catch (Exception e) {
                LOG.error(
                    "Failed to check if there are drafts older than {} in tenant '{}'. The draft GC will continue to run.",
                    ageThreshold,
                    tenant,
                    e);
                return true;
              }
            });
  }
}
