/*
 * © 2023-2025 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.impl.draft;

import com.sap.cds.impl.util.Stack;
import com.sap.cds.ql.CQL;
import com.sap.cds.ql.RefBuilder;
import com.sap.cds.ql.RefBuilder.RefSegment;
import com.sap.cds.ql.StructuredTypeRef;
import com.sap.cds.ql.cqn.CqnAnalyzer;
import com.sap.cds.ql.cqn.CqnArithmeticExpression;
import com.sap.cds.ql.cqn.CqnArithmeticNegation;
import com.sap.cds.ql.cqn.CqnBetweenPredicate;
import com.sap.cds.ql.cqn.CqnComparisonPredicate;
import com.sap.cds.ql.cqn.CqnComparisonPredicate.Operator;
import com.sap.cds.ql.cqn.CqnConnectivePredicate;
import com.sap.cds.ql.cqn.CqnContainmentTest;
import com.sap.cds.ql.cqn.CqnElementRef;
import com.sap.cds.ql.cqn.CqnExistsSubquery;
import com.sap.cds.ql.cqn.CqnExpand;
import com.sap.cds.ql.cqn.CqnExpression;
import com.sap.cds.ql.cqn.CqnFunc;
import com.sap.cds.ql.cqn.CqnInPredicate;
import com.sap.cds.ql.cqn.CqnInline;
import com.sap.cds.ql.cqn.CqnListValue;
import com.sap.cds.ql.cqn.CqnLiteral;
import com.sap.cds.ql.cqn.CqnMatchPredicate;
import com.sap.cds.ql.cqn.CqnNegation;
import com.sap.cds.ql.cqn.CqnNullValue;
import com.sap.cds.ql.cqn.CqnParameter;
import com.sap.cds.ql.cqn.CqnPassThroughSearchPredicate;
import com.sap.cds.ql.cqn.CqnPlain;
import com.sap.cds.ql.cqn.CqnPredicate;
import com.sap.cds.ql.cqn.CqnReference.Segment;
import com.sap.cds.ql.cqn.CqnSearchTermPredicate;
import com.sap.cds.ql.cqn.CqnSelect;
import com.sap.cds.ql.cqn.CqnSelectListValue;
import com.sap.cds.ql.cqn.CqnSortSpecification;
import com.sap.cds.ql.cqn.CqnStructuredTypeRef;
import com.sap.cds.ql.cqn.CqnToken;
import com.sap.cds.ql.cqn.CqnValue;
import com.sap.cds.ql.cqn.CqnVisitor;
import com.sap.cds.ql.cqn.ResolvedSegment;
import com.sap.cds.ql.impl.Xpr;
import com.sap.cds.reflect.CdsAssociationType;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsModel;
import com.sap.cds.reflect.CdsStructuredType;
import com.sap.cds.services.draft.DraftAdministrativeData;
import com.sap.cds.services.draft.Drafts;
import com.sap.cds.services.utils.DraftUtils;
import com.sap.cds.util.CdsModelUtils;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

public class DraftScenarioAnalyzer {

  public record AnalysisResult(
      Scenario scenario, CqnPredicate predicate, CqnStructuredTypeRef ref) {}

  public static AnalysisResult analyze(CqnSelect select, CdsEntity target, CdsModel model) {
    if (isAnalyticalQueryOnActives(select) || !referencesDrafts(select, target, model)) {
      return new AnalysisResult(Scenario.ALL_HIDING_DRAFTS, null, null);
    } else if (select.groupBy().isEmpty()
        && select.having().isEmpty()
        && StatementAnalyzer.analyze(select, target).isOptimizable()
        && PredicateAnalyzer.analyze(select.search()).getComparison() == Comparison.OTHER) {

      CqnStructuredTypeRef ref = select.ref();
      RefAnalyzer refAnalyzer = RefAnalyzer.analyze(ref, model);
      PredicateAnalyzer whereAnalyzer = PredicateAnalyzer.analyze(select.where());

      Scenario scenario =
          mergeScenarios(refAnalyzer.getScenario(), whereAnalyzer.getScenario(), ref);
      return new AnalysisResult(
          scenario, whereAnalyzer.getPredicate(), refAnalyzer.getRemainingRef());
    }
    return new AnalysisResult(Scenario.NOT_OPTIMIZABLE, null, null);
  }

  private static boolean isAnalyticalQueryOnActives(CqnSelect select) {
    // Select from Select => Always assume execution on Actives only
    if (!select.from().isRef()) {
      return true;
    }
    // Analytics or TreeTable
    if (!select.transformations().isEmpty()) {
      CqnStructuredTypeRef ref = select.from().asRef();
      // statements targeting a root (draft-enabled) entity directly
      // are only executed on actives (we never merge actives & inactives for these requests)
      // statements with a ref > 2 or a filter in the first segment can be analyzed
      // they will be pinned to either actives or inactives by analysis
      return ref.size() == 1 && !ref.rootSegment().filter().isPresent();
    }
    return false;
  }

  private static boolean referencesDrafts(CqnSelect select, CdsEntity target, CdsModel model) {
    if (DraftUtils.isDraftEnabled(target)
        || target.getName().equals(Drafts.DRAFT_ADMINISTRATIVE_DATA)) {
      return true; // fast path
    }
    Iterator<ResolvedSegment> iterator = CqnAnalyzer.create(model).analyze(select.ref()).iterator();
    while (iterator.hasNext()) {
      if (DraftUtils.isDraftEnabled(iterator.next().type())) {
        return true;
      }
    }
    return false;
  }

  private static Scenario mergeScenarios(
      Scenario refScenario, Scenario whereScenario, CqnStructuredTypeRef ref) {
    Scenario scenario;
    if (refScenario == Scenario.OTHER) {
      scenario = whereScenario;
    } else if ((refScenario == Scenario.ALL_HIDING_DRAFTS
            && (whereScenario == Scenario.ALL_HIDING_DRAFTS || whereScenario == Scenario.OTHER))
        || (refScenario == Scenario.OWN_DRAFT
            && (whereScenario == Scenario.OWN_DRAFT || whereScenario == Scenario.OTHER))) {
      String lastSegment = ref.lastSegment();
      if (lastSegment.equals(Drafts.SIBLING_ENTITY)) {
        // flip scenario in case of SiblingEntity navigation
        scenario =
            refScenario == Scenario.OWN_DRAFT ? Scenario.ALL_HIDING_DRAFTS : Scenario.OWN_DRAFT;
      } else if (lastSegment.equals(Drafts.DRAFT_ADMINISTRATIVE_DATA)) {
        // always read DraftAdministrativeData via draft
        scenario =
            refScenario == Scenario.OWN_DRAFT
                ? Scenario.OWN_DRAFT
                : Scenario.DRAFT_ADMINISTRATIVE_DATA_VIA_ACTIVE;
      } else {
        scenario = refScenario;
      }
    } else {
      scenario = Scenario.NOT_OPTIMIZABLE;
    }
    return scenario;
  }

  private static class StatementAnalyzer implements CqnVisitor {

    private static StatementAnalyzer analyze(CqnSelect select, CdsEntity target) {
      StatementAnalyzer validator = new StatementAnalyzer(target, false);
      select.accept(validator);
      return validator;
    }

    private final CdsEntity entity;
    private final Set<String> keyNames;
    private final boolean forExpand;

    private boolean optimizable = true;

    public StatementAnalyzer(CdsEntity entity, boolean forExpand) {
      this.entity = entity;
      this.keyNames = entity.keyElements().map(CdsElement::getName).collect(Collectors.toSet());
      this.forExpand = forExpand;
    }

    public boolean isOptimizable() {
      return optimizable;
    }

    @Override
    public void visit(CqnStructuredTypeRef typeRef) {
      if (optimizable
          && ((!forExpand
                  && typeRef.segments().subList(0, typeRef.size() - 1).stream()
                      .anyMatch(s -> s.id().equals(Drafts.SIBLING_ENTITY)))
              || // sibling entity navigation only at the very end
              (forExpand
                  && (typeRef.size() != 1
                      || typeRef.rootSegment().filter().isPresent()
                      || typeRef
                          .rootSegment()
                          .id()
                          .equals(Drafts.SIBLING_ENTITY))) // no multi segment expands or filters in
          // expands and no SiblingEntity expand
          )) {
        optimizable = false;
      }
    }

    @Override
    public void visit(CqnSortSpecification sortSpec) {
      // orderby must not contain draft elements, except IsActiveEntity
      if (optimizable
          && sortSpec
              .value()
              .ofRef()
              .filter(r -> !r.path().equals(Drafts.IS_ACTIVE_ENTITY))
              .anyMatch(DraftScenarioAnalyzer::containsDraftElements)) {
        optimizable = false;
      }
    }

    @Override
    public void visit(CqnSelectListValue slv) {
      if (optimizable) {
        CqnValue value = slv.value();
        String elementName = null;
        if (value.isRef()) {
          // no multi-segment references to draft elements
          CqnElementRef ref = value.asRef();
          if (ref.size() > 1 && containsDraftElements(ref)) {
            optimizable = false;
          } else {
            elementName = ref.path();
          }
        } else if (value.isFunction()) {
          if (value.asFunction().args().stream()
              .flatMap(v -> v.ofRef())
              .anyMatch(DraftScenarioAnalyzer::containsDraftElements)) {
            optimizable = false;
          }
        } else if (refsOf(value).anyMatch(DraftScenarioAnalyzer::containsDraftElements)) {
          optimizable = false;
        }

        if (optimizable
            && slv.alias().isPresent()
            && !slv.alias().get().equals(elementName)
            && (keyNames.contains(slv.alias().get())
                || (elementName != null && Drafts.ELEMENTS.contains(elementName)))) {
          optimizable = false;
        }
      }
    }

    @Override
    public void visit(CqnExpand expand) {
      if (optimizable) {
        if (expand.ref().path().equals("*")) {
          optimizable = false;
        } else {
          StatementAnalyzer analyzer =
              new StatementAnalyzer(CdsModelUtils.entity(entity, expand.ref().segments()), true);
          expand.ref().accept(analyzer);
          expand.items().forEach(i -> i.accept(analyzer));
          expand.orderBy().forEach(o -> o.accept(analyzer));
          optimizable = analyzer.isOptimizable();
        }
      }
    }

    @Override
    public void visit(CqnInline inline) {
      optimizable = false;
    }

    private Stream<CqnElementRef> refsOf(CqnToken token) {
      return token.tokens().filter(CqnElementRef.class::isInstance).map(CqnElementRef.class::cast);
    }
  }

  public enum Scenario {
    ALL, // IsActiveEntity eq false or SiblingEntity/IsActiveEntity eq null
    ALL_HIDING_DRAFTS, // IsActiveEntity eq true
    UNCHANGED, // IsActiveEntity eq true and HasDraftEntity eq false
    OWN_DRAFT, // IsActiveEntity eq false
    LOCKED_BY_ANOTHER_USER, // IsActiveEntity eq true and SiblingEntity/IsActiveEntity eq null and
    // DraftAdministrativeData/InProcessByUser ne '' and
    // DraftAdministrativeData/InProcessByUser ne null
    UNSAVED_CHANGES_BY_ANOTHER_USER, // IsActiveEntity eq true and SiblingEntity/IsActiveEntity eq
    // null and DraftAdministrativeData/InProcessByUser eq ''
    DRAFT_ADMINISTRATIVE_DATA_VIA_ACTIVE, // no Fiori draft scenario, internal marker
    OTHER, // no Fiori draft scenario, but optimizable -> arbitrary filter only touching
    // non-draft-related elements
    NOT_OPTIMIZABLE
  }

  private enum Comparison {
    IS_ACTIVE_ENTITY_EQ_TRUE,
    IS_ACTIVE_ENTITY_EQ_FALSE,
    SIBLING_ENTITY_IS_ACTIVE_ENTITY_EQ_NULL,
    HAS_DRAFT_ENTITY_EQ_FALSE,
    DRAFT_ADMINISTRATIVE_DATA_IN_PROCESS_BY_USER_EQ_EMPTY,
    DRAFT_ADMINISTRATIVE_DATA_IN_PROCESS_BY_USER_NE_EMPTY,
    DRAFT_ADMINISTRATIVE_DATA_IN_PROCESS_BY_USER_NE_NULL,
    DRAFT_RELATED,
    OTHER // arbitrary filter only touching non-draft-related elements
  }

  private static class PredicateAnalyzer implements CqnVisitor {

    private static PredicateAnalyzer analyze(Optional<CqnPredicate> predicate) {
      PredicateAnalyzer analyzer = new PredicateAnalyzer();
      predicate.ifPresent(p -> p.accept(analyzer));
      return analyzer;
    }

    private static final String IS_ACTIVE_ENTITY = Drafts.IS_ACTIVE_ENTITY;
    private static final String SIBLING_ENTITY_IS_ACTIVE_ENTITY =
        Drafts.SIBLING_ENTITY + "." + Drafts.IS_ACTIVE_ENTITY;
    private static final String HAS_DRAFT_ENTITY = Drafts.HAS_DRAFT_ENTITY;
    private static final String DRAFT_ADMINISTRATIVE_DATA_IN_PROCESS_BY_USER =
        Drafts.DRAFT_ADMINISTRATIVE_DATA + "." + DraftAdministrativeData.IN_PROCESS_BY_USER;

    private final Stack<Object> scenarioStack = new Stack<>();
    private final Stack<CqnToken> predicateStack = new Stack<>();

    public Scenario getScenario() {
      Object comparison = getComparison();
      if (comparison == Comparison.IS_ACTIVE_ENTITY_EQ_TRUE) {
        comparison = Scenario.ALL_HIDING_DRAFTS;
      } else if (comparison == Comparison.IS_ACTIVE_ENTITY_EQ_FALSE) {
        comparison = Scenario.OWN_DRAFT;
      } else if (comparison == Comparison.OTHER) {
        comparison = Scenario.OTHER;
      }
      return comparison instanceof Scenario s ? s : Scenario.NOT_OPTIMIZABLE;
    }

    public CqnPredicate getPredicate() {
      CqnToken token = predicateStack.size() == 1 ? predicateStack.pop() : CQL.TRUE;
      return token instanceof CqnPredicate p ? p : CQL.TRUE;
    }

    private Object getComparison() {
      return scenarioStack.size() == 1 ? scenarioStack.pop() : Comparison.OTHER;
    }

    @Override
    public void visit(CqnComparisonPredicate c) {
      Object comparison = comparisonType(scenarioStack.pop(2));
      String ref =
          c.left().isRef()
              ? c.left().asRef().path()
              : c.right().isRef() ? c.right().asRef().path() : null;
      // one side must be a ref
      if (ref != null) {
        boolean isNull = c.left().isNullValue() || c.right().isNullValue();
        Object value =
            c.left().isLiteral()
                ? c.left().asLiteral().value()
                : c.right().isLiteral() ? c.right().asLiteral().value() : null;
        // compared with null or some literal value
        if (isNull || value != null) {
          var op = c.operator();
          if (ref.equals(IS_ACTIVE_ENTITY) && (op == Operator.IS || op == Operator.EQ)) {
            if (Boolean.TRUE.equals(value)) {
              comparison = Comparison.IS_ACTIVE_ENTITY_EQ_TRUE;
            } else if (Boolean.FALSE.equals(value)) {
              comparison = Comparison.IS_ACTIVE_ENTITY_EQ_FALSE;
            }
          } else if (ref.equals(SIBLING_ENTITY_IS_ACTIVE_ENTITY) && op == Operator.IS && isNull) {
            comparison = Comparison.SIBLING_ENTITY_IS_ACTIVE_ENTITY_EQ_NULL;
          } else if (ref.equals(HAS_DRAFT_ENTITY)
              && (op == Operator.IS || op == Operator.EQ)
              && Boolean.FALSE.equals(value)) {
            comparison = Comparison.HAS_DRAFT_ENTITY_EQ_FALSE;
          } else if (ref.equals(DRAFT_ADMINISTRATIVE_DATA_IN_PROCESS_BY_USER)) {
            if ("".equals(value)) {
              if (op == Operator.IS || op == Operator.EQ) {
                comparison = Comparison.DRAFT_ADMINISTRATIVE_DATA_IN_PROCESS_BY_USER_EQ_EMPTY;
              } else if (op == Operator.IS_NOT || op == Operator.NE) {
                comparison = Comparison.DRAFT_ADMINISTRATIVE_DATA_IN_PROCESS_BY_USER_NE_EMPTY;
              }
            } else if (op == Operator.IS_NOT && isNull) {
              comparison = Comparison.DRAFT_ADMINISTRATIVE_DATA_IN_PROCESS_BY_USER_NE_NULL;
            }
          }
        }
      }
      scenarioStack.push(comparison);
      predicateStack.pop(2);
      // replace scenario comparisons with TRUE
      if (comparison == Comparison.OTHER || comparison == Comparison.DRAFT_RELATED) {
        predicateStack.push(c);
      } else {
        predicateStack.push(CQL.TRUE);
      }
    }

    @Override
    public void visit(CqnConnectivePredicate connective) {
      Set<Object> comparisons = new HashSet<>(scenarioStack.pop(connective.predicates().size()));
      // a draft scenario connected with AND with other non draft scenarios or comparisons, remains
      // the draft scenario
      // therefore we can ignore Comparison.OTHER if the Operator is AND
      if (connective.operator() == CqnConnectivePredicate.Operator.AND) {
        comparisons.remove(Comparison.OTHER);
      }
      if (comparisons.size() == 2
          && connective.operator() == CqnConnectivePredicate.Operator.OR
          && comparisons.contains(Comparison.IS_ACTIVE_ENTITY_EQ_FALSE)
          && comparisons.contains(Comparison.SIBLING_ENTITY_IS_ACTIVE_ENTITY_EQ_NULL)) {
        scenarioStack.push(Scenario.ALL);
      } else if (comparisons.size() == 2
          && connective.operator() == CqnConnectivePredicate.Operator.AND
          && comparisons.contains(Comparison.IS_ACTIVE_ENTITY_EQ_TRUE)
          && comparisons.contains(Comparison.HAS_DRAFT_ENTITY_EQ_FALSE)) {
        scenarioStack.push(Scenario.UNCHANGED);
      } else if (comparisons.size() == 3
          && connective.operator() == CqnConnectivePredicate.Operator.AND
          && comparisons.contains(Comparison.IS_ACTIVE_ENTITY_EQ_TRUE)
          && comparisons.contains(Comparison.SIBLING_ENTITY_IS_ACTIVE_ENTITY_EQ_NULL)
          && comparisons.contains(
              Comparison.DRAFT_ADMINISTRATIVE_DATA_IN_PROCESS_BY_USER_EQ_EMPTY)) {
        scenarioStack.push(Scenario.UNSAVED_CHANGES_BY_ANOTHER_USER);
      } else if (comparisons.size() == 4
          && connective.operator() == CqnConnectivePredicate.Operator.AND
          && comparisons.contains(Comparison.IS_ACTIVE_ENTITY_EQ_TRUE)
          && comparisons.contains(Comparison.SIBLING_ENTITY_IS_ACTIVE_ENTITY_EQ_NULL)
          && comparisons.contains(Comparison.DRAFT_ADMINISTRATIVE_DATA_IN_PROCESS_BY_USER_NE_EMPTY)
          && comparisons.contains(
              Comparison.DRAFT_ADMINISTRATIVE_DATA_IN_PROCESS_BY_USER_NE_NULL)) {
        scenarioStack.push(Scenario.LOCKED_BY_ANOTHER_USER);
      } else if (comparisons.size() == 1
          && connective.operator() == CqnConnectivePredicate.Operator.AND
          && (comparisons.iterator().next() instanceof Scenario
              || comparisons.contains(Comparison.IS_ACTIVE_ENTITY_EQ_TRUE)
              || comparisons.contains(Comparison.IS_ACTIVE_ENTITY_EQ_FALSE))) {
        scenarioStack.push(comparisons.iterator().next());
      } else {
        scenarioStack.push(comparisonType(comparisons));
      }
      // merge potentially replaced predicates
      List<CqnToken> predicateTokens = predicateStack.pop(connective.predicates().size());
      predicateStack.push(
          CQL.connect(
              connective.operator(), predicateTokens.stream().map(t -> (CqnPredicate) t).toList()));
    }

    @Override
    public void visit(CqnElementRef elementRef) {
      if (containsDraftElements(elementRef)) {
        scenarioStack.push(Comparison.DRAFT_RELATED);
      } else {
        scenarioStack.push(Comparison.OTHER);
      }
      predicateStack.push(elementRef);
    }

    @Override
    public void visit(CqnPlain plain) {
      scenarioStack.push(Comparison.DRAFT_RELATED); // potentially
      predicateStack.push(plain);
    }

    @Override
    public void visit(CqnParameter param) {
      pushOther(param);
    }

    @Override
    public void visit(CqnLiteral<?> literal) {
      pushOther(literal);
    }

    @Override
    public void visit(CqnNullValue nil) {
      pushOther(nil);
    }

    @Override
    public void visit(CqnExistsSubquery exists) {
      pushOther(exists);
    }

    @Override
    public void visit(CqnMatchPredicate match) {
      pushOther(match);
    }

    @Override
    public void visit(CqnFunc func) {
      popPush(func.args().size(), func);
    }

    @Override
    public void visit(CqnListValue list) {
      popPush(list.size(), list);
    }

    @Override
    public void visit(CqnSearchTermPredicate search) {
      pushOther(search);
    }

    @Override
    public void visit(CqnPassThroughSearchPredicate search) {
      pushOther(search);
    }

    @Override
    public void visit(CqnContainmentTest test) {
      popPush(2, test);
    }

    @Override
    public void visit(CqnInPredicate in) {
      popPush(2, in);
    }

    @Override
    public void visit(CqnArithmeticExpression expr) {
      popPush(2, expr);
    }

    @Override
    public void visit(CqnArithmeticNegation neg) {
      popPush(1, neg);
    }

    @Override
    public void visit(CqnNegation neg) {
      popPush(1, neg);
    }

    @Override
    public void visit(CqnExpression expr) {
      popPush(((Xpr) expr).size(), expr);
    }

    @Override
    public void visit(CqnBetweenPredicate between) {
      popPush(3, between);
    }

    private Object comparisonType(Collection<Object> comparisons) {
      return comparisons.stream().allMatch(Comparison.OTHER::equals)
          ? Comparison.OTHER
          : Comparison.DRAFT_RELATED;
    }

    private void pushOther(CqnToken token) {
      scenarioStack.push(Comparison.OTHER);
      predicateStack.push(token);
    }

    private void popPush(int pop, CqnToken token) {
      scenarioStack.push(comparisonType(scenarioStack.pop(pop)));
      predicateStack.pop(pop);
      predicateStack.push(token);
    }
  }

  private static boolean containsDraftElements(CqnElementRef ref) {
    return ref.segments().stream().map(Segment::id).anyMatch(Drafts.ELEMENTS::contains);
  }

  private static class RefAnalyzer {

    private final CdsModel model;
    private CqnStructuredTypeRef remainingRef;
    private Scenario scenario = Scenario.NOT_OPTIMIZABLE;

    public static RefAnalyzer analyze(CqnStructuredTypeRef ref, CdsModel model) {
      RefAnalyzer analyzer = new RefAnalyzer(ref, model);
      analyzer.analyze();
      return analyzer;
    }

    private RefAnalyzer(CqnStructuredTypeRef ref, CdsModel model) {
      this.model = model;
      this.remainingRef = ref;
    }

    public Scenario getScenario() {
      return scenario;
    }

    public CqnStructuredTypeRef getRemainingRef() {
      return remainingRef;
    }

    private void analyze() {
      // copy remainingRef for modification, we iterate over segments of newRef
      RefBuilder<StructuredTypeRef> newRef = CQL.copy(remainingRef);
      PredicateAnalyzer rootAnalyzer = PredicateAnalyzer.analyze(newRef.rootSegment().filter());
      Scenario scenario = rootAnalyzer.getScenario();
      CdsStructuredType currentType = model.getEntity(newRef.rootSegment().id());
      boolean currentIsDraft = DraftUtils.isDraftEnabled(currentType);
      if (scenario == Scenario.ALL_HIDING_DRAFTS
          || scenario == Scenario.OWN_DRAFT
          || !currentIsDraft) {
        newRef.rootSegment().filter(rootAnalyzer.getPredicate());

        int endIndex = newRef.segments().size();
        if (newRef.targetSegment().id().equals(Drafts.SIBLING_ENTITY)) {
          newRef.segments().remove(--endIndex); // remove SIBLING_ENTITY
        } else if (newRef.targetSegment().id().equals(Drafts.DRAFT_ADMINISTRATIVE_DATA)) {
          --endIndex; // ignore DRAFT_ADMINISTRATIVE_DATA
        }

        CdsStructuredType parentType = currentType;
        boolean parentIsDraft = currentIsDraft;
        for (int i = 1; i < endIndex; ++i) {
          RefSegment segment = newRef.segments().get(i);
          currentType = parentType.getTargetOf(segment.id());
          currentIsDraft =
              parentType instanceof CdsEntity
                  ? DraftUtils.isDraftEnabled(currentType)
                  : parentIsDraft;

          PredicateAnalyzer segmentAnalyzer = PredicateAnalyzer.analyze(segment.filter());
          Scenario segmentScenario = segmentAnalyzer.getScenario();
          // TODO composition is not the right indicator for the same draft document
          // also consider backlink associations and associations with @odata.draft.enclosed
          boolean isWithinDocument =
              parentType
                  .findAssociation(segment.id())
                  .map(a -> a.getType().as(CdsAssociationType.class).isComposition())
                  .orElse(!(currentType instanceof CdsEntity));

          if (isStepToSameDraftDocument(
                  parentIsDraft, isWithinDocument, currentIsDraft, segmentScenario, scenario)
              || isStepToActiveDraft(currentIsDraft, segmentScenario, scenario)
              || isStepToNonDraft(currentIsDraft, segmentScenario, scenario)) {
            if (scenario == Scenario.OTHER && segmentScenario == Scenario.ALL_HIDING_DRAFTS) {
              scenario = Scenario.ALL_HIDING_DRAFTS;
            }
            segment.filter(segmentAnalyzer.getPredicate());
            parentType = currentType;
            parentIsDraft = currentIsDraft;
          } else {
            scenario = Scenario.NOT_OPTIMIZABLE;
            return;
          }
        }
      }
      this.scenario = scenario;
      this.remainingRef = newRef.build();
    }

    private boolean isStepToSameDraftDocument(
        boolean parentIsDraft,
        boolean isWithinDocument,
        boolean currentIsDraft,
        Scenario segmentScenario,
        Scenario scenario) {
      // draft -(comp)-> draft => keeps scenario
      return parentIsDraft
          && isWithinDocument
          && currentIsDraft
          && (segmentScenario == scenario || segmentScenario == Scenario.OTHER);
    }

    private boolean isStepToActiveDraft(
        boolean currentIsDraft, Scenario segmentScenario, Scenario scenario) {
      // draft/non-draft --> other draft => active might navigate to other active
      return currentIsDraft
          && (segmentScenario == Scenario.ALL_HIDING_DRAFTS || segmentScenario == Scenario.OTHER)
          && (scenario == Scenario.ALL_HIDING_DRAFTS || scenario == Scenario.OTHER);
    }

    private boolean isStepToNonDraft(
        boolean currentIsDraft, Scenario segmentScenario, Scenario scenario) {
      // draft/non-draft --> non-draft => keeps scenario
      return !currentIsDraft && segmentScenario == Scenario.OTHER;
    }
  }
}
