/*
 * © 2019-2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.impl.environment;

import com.sap.cds.services.utils.StringUtils;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Scanner;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class DefaultEnvFileUtils {

  private static final Logger logger = LoggerFactory.getLogger(DefaultEnvFileUtils.class);
  private static final String DEFAULT_ENV = "default-env.json";
  private static final String CLASSPATH = "classpath:";

  /**
   * @return the value of the default-env.json file as String.
   */
  static String getDefaultEnv(String configuredPath) {
    // look at the configured path
    InputStream stream = getFromDefaultEnvPath(configuredPath);

    // try to read default-env.json from classpath resources
    if (stream == null) {
      stream = getClassLoaderResource(DEFAULT_ENV);
    }

    // now try looking for default-env.json inside the working directory
    Path workingDirectory = Paths.get(System.getProperty("user.dir"));

    if (stream == null) {
      stream = getFromDirectory(workingDirectory, DEFAULT_ENV);
    }

    // try within parent of working directory, parent could be null in docker container
    if (stream == null && workingDirectory.getParent() != null) {
      stream = getFromDirectory(workingDirectory.getParent(), DEFAULT_ENV);
    }

    if (stream != null) {
      try (Scanner scanner = new Scanner(stream, StandardCharsets.UTF_8.name())) {
        return scanner.useDelimiter("\\A").next();
      }
    }
    return null;
  }

  private static InputStream getFromDirectory(Path workingDirectory, String fileName) {
    File defaultEnvFile = workingDirectory.resolve(fileName).toFile();
    if (defaultEnvFile.exists()) {
      try {
        logger.info(
            "Loaded {} from directory '{}'", defaultEnvFile.getName(), defaultEnvFile.getParent());
        return new FileInputStream(defaultEnvFile);
      } catch (FileNotFoundException e) { // NOSONAR
        // ignore
      }
    }
    return null;
  }

  private static InputStream getFromDefaultEnvPath(String defaultEnvPath) {
    if (StringUtils.isEmpty(defaultEnvPath)) {
      return null;
    }

    // try to read it as classloader resource
    if (defaultEnvPath.startsWith(CLASSPATH)) {
      return getClassLoaderResource(defaultEnvPath.substring(CLASSPATH.length()));
    }

    // check, if the path exists and is a file or directory
    File defaultEnvFile = new File(defaultEnvPath);
    if (defaultEnvFile.isDirectory()) {
      // configured as directory, use default file name
      defaultEnvFile = Paths.get(defaultEnvPath).resolve(DEFAULT_ENV).toFile();
    }
    if (defaultEnvFile.exists()) {
      try {
        logger.info(
            "Loaded {} from directory '{}'", defaultEnvFile.getName(), defaultEnvFile.getParent());
        return new FileInputStream(defaultEnvFile);
      } catch (FileNotFoundException e) { // NOSONAR
        // ignore
      }
    }

    logger.warn("No default env file found in configured path '{}'", defaultEnvPath);
    return null;
  }

  private static InputStream getClassLoaderResource(String resourceName) {
    InputStream stream =
        Thread.currentThread().getContextClassLoader().getResourceAsStream(resourceName);
    if (stream != null) {
      logger.info("Loaded {} from classpath resources", resourceName);
    }
    return stream;
  }
}
