/*
 * © 2019-2025 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.impl.handlerregistry;

import com.sap.cds.services.EventContext;
import com.sap.cds.services.Service;
import com.sap.cds.services.ServiceCatalog;
import com.sap.cds.services.handler.Handler;
import com.sap.cds.services.impl.handlerregistry.resolver.ArgumentResolver;
import com.sap.cds.services.outbox.OutboxService;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.StringUtils;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.function.Supplier;
import java.util.stream.Stream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HandlerRegistryTools {

  private static final Logger logger = LoggerFactory.getLogger(HandlerRegistryTools.class);

  private HandlerRegistryTools() {}

  public static <T> void registerInstance(Object instance, Service service) {
    AnnotatedHandlerMethodsFinder ahmf = new AnnotatedHandlerMethodsFinder(instance.getClass());

    for (HandlerDescriptor hd : ahmf.getHandlerDescriptors()) {
      for (AnnotationDescriptor ad : hd.getAnnotations()) {
        register(ad, hd, () -> instance, Stream.of(service), false);
      }
    }
  }

  public static <T> void registerClass(
      Class<T> handlerClass, Supplier<T> factory, ServiceCatalog serviceCatalog) {
    AnnotatedHandlerMethodsFinder ahmf = new AnnotatedHandlerMethodsFinder(handlerClass);

    for (HandlerDescriptor hd : ahmf.getHandlerDescriptors()) {
      for (AnnotationDescriptor ad : hd.getAnnotations()) {
        register(ad, hd, factory, serviceCatalog.getServices(), true);
      }
    }
  }

  private static <T> void register(
      AnnotationDescriptor ad,
      HandlerDescriptor hd,
      Supplier<T> factory,
      Stream<Service> availableServices,
      boolean requireServiceConfig) {
    Collection<String> names;
    String[] theDefaultNames = hd.getDefaultServiceNames();
    String[] theNames = ad.getServiceNames();
    if (StringUtils.notEmpty(theNames) != null) {
      names = EventPredicateTools.toMatchCollection(theNames);
    } else if (StringUtils.notEmpty(theDefaultNames) != null) {
      names = EventPredicateTools.toMatchCollection(theDefaultNames);
    } else {
      if (requireServiceConfig) {
        throw new ErrorStatusException(
            CdsErrorStatuses.HANDLER_SERVICE_REQUIRED, hd.getMethodName());
      } else {
        names = Collections.emptyList();
      }
    }

    Collection<Class<?>> types;
    Class<?>[] theDefaultTypes = hd.getDefaultServiceTypes();
    Class<?>[] theTypes = ad.getServiceTypes();
    if (theTypes.length > 0) {
      types = Arrays.asList(theTypes);
    } else if (theDefaultTypes.length > 0) {
      types = Arrays.asList(theDefaultTypes);
    } else {
      types = Collections.emptyList();
    }

    Stream<Service> theServices = availableServices.map(OutboxService::unboxed);
    if (!names.isEmpty()) {
      theServices = theServices.filter(s -> names.contains(s.getName()));
    }
    if (!types.isEmpty()) {
      theServices =
          theServices.filter(s -> types.stream().anyMatch(t -> t.isAssignableFrom(s.getClass())));
    }
    List<Service> services = theServices.toList();
    if (services.isEmpty()) {
      logger.warn(
          "Failed to register handler method '"
              + hd.getMethodName()
              + "': Could not find any matching service.");
      return;
    }

    String[] events;
    if (StringUtils.notEmpty(ad.getEvents()) == null) {
      // infer from argument resolvers
      Set<String> eventSet = new HashSet<>();
      for (ArgumentResolver resolver : hd.getArgumentResolvers()) {
        eventSet.addAll(Arrays.asList(resolver.indicateEvents()));
      }
      events = eventSet.toArray(new String[0]);

      // fail if no events were provided
      if (StringUtils.notEmpty(events) == null) {
        throw new ErrorStatusException(CdsErrorStatuses.HANDLER_EVENT_REQUIRED, hd.getMethodName());
      }
    } else {
      events = ad.getEvents();
    }

    String[] entities;
    if (StringUtils.notEmpty(ad.getEntities()) == null) {
      // infer from argument resolvers
      Set<String> entity = new HashSet<>();
      for (ArgumentResolver resolver : hd.getArgumentResolvers()) {
        entity.addAll(Arrays.asList(resolver.indicateEntities()));
      }
      entities = entity.toArray(new String[0]);
    } else {
      entities = ad.getEntities();
    }

    services.forEach(
        (service) -> {
          switch (ad.getPhase()) {
            case BEFORE:
              service.before(events, entities, hd.getOrder(), new DescribedHandler<T>(hd, factory));
              break;
            case ON:
              service.on(events, entities, hd.getOrder(), new DescribedHandler<T>(hd, factory));
              break;
            case AFTER:
              service.after(events, entities, hd.getOrder(), new DescribedHandler<T>(hd, factory));
              break;
            default:
              // do not handle
          }
        });
  }

  private static final class DescribedHandler<T> implements Handler {

    private final HandlerDescriptor hd;
    private final Supplier<T> factory;

    DescribedHandler(HandlerDescriptor hd, Supplier<T> factory) {
      this.hd = hd;
      this.factory = factory;
    }

    @Override
    public void process(EventContext context) {
      try {
        // transform arguments
        List<ArgumentResolver> argumentResolvers = hd.getArgumentResolvers();
        Object[] arguments = new Object[argumentResolvers.size() + 1];
        // first argument of the invoke method, needs to be the object on which the method is called
        arguments[0] = factory.get();

        for (int i = 0; i < argumentResolvers.size(); i++) {
          arguments[i + 1] = argumentResolvers.get(i).resolve(context);
        }

        // when the method handle was created it was already enabled as a spreader
        Object returnValue = hd.getMethodHandle().invoke(arguments);

        // transform return value
        hd.getReturnResolver().resolve(returnValue, context);
      } catch (Throwable e) { // NOSONAR
        throw throwAsUnchecked(e);
      }
    }

    @Override
    public String toString() {
      return hd.getMethodName();
    }
  }

  private static RuntimeException throwAsUnchecked(Throwable throwable) {
    HandlerRegistryTools.<RuntimeException>_throwAsUnchecked(throwable);
    return null;
  }

  @SuppressWarnings({"unchecked"})
  private static <T extends Throwable> void _throwAsUnchecked(Throwable throwable) throws T {
    throw (T) throwable;
  }
}
