/*
 * © 2025 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.impl.handlerregistry.resolver;

import com.sap.cds.impl.builder.model.StructuredTypeImpl;
import com.sap.cds.ql.CQL;
import com.sap.cds.ql.CdsName;
import com.sap.cds.ql.StructuredType;
import com.sap.cds.ql.cqn.CqnStatement;
import com.sap.cds.ql.cqn.CqnStructuredTypeRef;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.impl.handlerregistry.HandlerDescriptor;

/**
 * Handles all types of {@link CqnStatement} and {@link CqnStructuredTypeRef} based arguments in
 * handler methods
 */
@SuppressWarnings({"rawtypes", "unchecked"})
public class CqnReferenceArgumentResolver implements ArgumentResolver {

  public static CqnReferenceArgumentResolver createIfApplicable(Class<?> type) {
    if (CqnStructuredTypeRef.class.equals(type)) {
      return new CqnReferenceArgumentResolver(null);
    } else if (StructuredType.class.isAssignableFrom(type)) {
      return new CqnReferenceArgumentResolver(type);
    }
    return null;
  }

  private final Class<? extends StructuredType> structuredTypeClass;

  private CqnReferenceArgumentResolver(Class<?> type) {
    this.structuredTypeClass = (Class<? extends StructuredType>) type;
  }

  @Override
  public Object resolve(EventContext context) {
    CqnStatement cqn = (CqnStatement) context.get("cqn");
    if (cqn != null && (!cqn.isSelect() || cqn.asSelect().from().isRef())) {
      CqnStructuredTypeRef ref = cqn.ref();
      if (structuredTypeClass != null) {
        if (StructuredType.class.equals(structuredTypeClass)) {
          return StructuredTypeImpl.structuredType(ref);
        } else {
          return CQL.entity(structuredTypeClass, ref);
        }
      } else {
        return ref;
      }
    } else {
      return null;
    }
  }

  @Override
  public String[] indicateEntities() {
    if (structuredTypeClass != null) {
      CdsName entityName = structuredTypeClass.getAnnotation(CdsName.class);
      if (entityName != null) {
        return new String[] {entityName.value()};
      }
    }
    return new String[0];
  }

  @Override
  public void verifyOrThrow(HandlerDescriptor descriptor) {
    // not yet validated
  }
}
