/*
 * © 2019-2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.impl.handlerregistry.resolver;

import com.google.common.collect.Sets;
import com.sap.cds.CdsData;
import com.sap.cds.CdsResult;
import com.sap.cds.Result;
import com.sap.cds.Row;
import com.sap.cds.ql.CdsName;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.cds.CdsCreateEventContext;
import com.sap.cds.services.cds.CdsDeleteEventContext;
import com.sap.cds.services.cds.CdsReadEventContext;
import com.sap.cds.services.cds.CdsUpdateEventContext;
import com.sap.cds.services.cds.CdsUpsertEventContext;
import com.sap.cds.services.cds.CqnService;
import com.sap.cds.services.draft.ActiveReadEventContext;
import com.sap.cds.services.draft.DraftCancelEventContext;
import com.sap.cds.services.draft.DraftCreateEventContext;
import com.sap.cds.services.draft.DraftEditEventContext;
import com.sap.cds.services.draft.DraftNewEventContext;
import com.sap.cds.services.draft.DraftPatchEventContext;
import com.sap.cds.services.draft.DraftPrepareEventContext;
import com.sap.cds.services.draft.DraftReadEventContext;
import com.sap.cds.services.draft.DraftSaveEventContext;
import com.sap.cds.services.draft.DraftService;
import com.sap.cds.services.impl.handlerregistry.AnnotationDescriptor;
import com.sap.cds.services.impl.handlerregistry.HandlerDescriptor;
import com.sap.cds.services.impl.utils.ReflectionUtils;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.ResultUtils;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Stream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Handles all types of CDS Model POJO based arguments in handler methods. It supports {@link List},
 * {@link Stream} and direct usage of POJOs
 */
public class PojoArgumentResolver implements ArgumentResolver {

  private static final Logger logger = LoggerFactory.getLogger(PojoArgumentResolver.class);
  private static final Set<String> allowedEvents =
      Sets.newHashSet(
          CqnService.EVENT_READ,
          CqnService.EVENT_CREATE,
          CqnService.EVENT_UPDATE,
          CqnService.EVENT_DELETE,
          CqnService.EVENT_UPSERT,
          DraftService.EVENT_DRAFT_EDIT,
          DraftService.EVENT_DRAFT_PREPARE,
          DraftService.EVENT_DRAFT_SAVE,
          DraftService.EVENT_DRAFT_NEW,
          DraftService.EVENT_DRAFT_PATCH,
          DraftService.EVENT_DRAFT_CANCEL,
          DraftService.EVENT_DRAFT_CREATE,
          DraftService.EVENT_DRAFT_READ,
          DraftService.EVENT_ACTIVE_READ);
  private static final HashMap<Type, PojoArgumentResolver> cache = new HashMap<>();

  private static enum ArgumentType {
    SINGLE,
    STREAM,
    LIST,
    RESULT,
    CDS_RESULT
  }

  /**
   * Returns a {@link PojoArgumentResolver}, if the class and generic type are applicable for this
   * resolver
   *
   * @param clazz the class of the argument
   * @param genericType the generic argument type
   * @return A {@link PojoArgumentResolver}, or null, if the resolver is not applicable.
   */
  public static PojoArgumentResolver createIfApplicable(Class<?> clazz, Type genericType) {
    PojoArgumentResolver cached;
    if ((cached = cache.get(genericType)) != null) {
      return cached;
    }

    try {
      ArgumentType argType = resolveArgumentType(clazz);
      if (argType != null) {
        Class<?> pojoClass = resolvePojoClass(clazz, genericType, argType);
        if (pojoClass != null) {
          return new PojoArgumentResolver(pojoClass, genericType, argType);
        }
      }
    } catch (IllegalArgumentException | ClassNotFoundException e) {
      logger.debug("Could not resolve generic type '{}'", genericType.getTypeName(), e);
    }

    return null;
  }

  private static ArgumentType resolveArgumentType(Class<?> clazz) {
    if (isPojoClass(clazz)) {
      return ArgumentType.SINGLE;
    } else if (Result.class.isAssignableFrom(clazz)) {
      return ArgumentType.RESULT;
    } else if (Stream.class.isAssignableFrom(clazz)) {
      return ArgumentType.STREAM;
    } else if (List.class.isAssignableFrom(clazz)) {
      return ArgumentType.LIST;
    } else if (CdsResult.class.isAssignableFrom(clazz)) {
      return ArgumentType.CDS_RESULT;
    }
    return null;
  }

  private static Class<?> resolvePojoClass(Class<?> clazz, Type genericType, ArgumentType argType)
      throws ClassNotFoundException {
    if (argType == ArgumentType.SINGLE) {
      return clazz;
    } else if (argType == ArgumentType.RESULT) {
      return Row.class;
    } else {
      ParameterizedType collectionType = ReflectionUtils.getParameterizedType(genericType);
      if (collectionType.getActualTypeArguments().length == 1) {
        Type pojoType = collectionType.getActualTypeArguments()[0];
        Class<?> pojoClass = ReflectionUtils.getClassForType(pojoType);
        if (isPojoClass(pojoClass)) {
          return pojoClass;
        } else if (argType == ArgumentType.CDS_RESULT && Object.class.equals(pojoClass)) {
          return CdsData.class; // cds result is typed at least with CdsData
        }
      }
      return null;
    }
  }

  /**
   * True, if the type is of a CSD Model POJO class.
   *
   * @param type the type
   * @return true, if the type is of a CDS Model POJO class.
   */
  private static boolean isPojoClass(Class<?> pojoClass) {
    return CdsData.class.isAssignableFrom(pojoClass);
  }

  private final Class<? extends CdsData> pojoClass;
  private final Type genericType;
  private final ArgumentType argumentType;

  @SuppressWarnings("unchecked")
  private PojoArgumentResolver(Class<?> pojoClass, Type genericType, ArgumentType argumentType) {
    this.pojoClass = (Class<? extends CdsData>) pojoClass;
    this.genericType = genericType;
    this.argumentType = argumentType;
    cache.put(genericType, this);
  }

  @Override
  public Object resolve(EventContext context) {
    if (context.isCompleted()) {
      // go to result in AFTER phase
      switch (context.getEvent()) {
        case CqnService.EVENT_READ:
          return convert(context.as(CdsReadEventContext.class).getResult());
        case CqnService.EVENT_CREATE:
          return convert(context.as(CdsCreateEventContext.class).getResult());
        case CqnService.EVENT_UPDATE:
          return convert(
              context.as(CdsUpdateEventContext.class).getResult()); // might be only partial
        case CqnService.EVENT_DELETE:
          return convert(
              context
                  .as(CdsDeleteEventContext.class)
                  .getResult()); // no data available by default, but custom ON handler might add it
        case CqnService.EVENT_UPSERT:
          return convert(context.as(CdsUpsertEventContext.class).getResult());
        case DraftService.EVENT_DRAFT_NEW:
          return convert(context.as(DraftNewEventContext.class).getResult());
        case DraftService.EVENT_DRAFT_EDIT:
          return convert(context.as(DraftEditEventContext.class).getResult());
        case DraftService.EVENT_DRAFT_PATCH:
          return convert(context.as(DraftPatchEventContext.class).getResult());
        case DraftService.EVENT_DRAFT_PREPARE:
          return convert(context.as(DraftPrepareEventContext.class).getResult());
        case DraftService.EVENT_DRAFT_SAVE:
          return convert(context.as(DraftSaveEventContext.class).getResult());
        case DraftService.EVENT_DRAFT_CANCEL:
          return convert(context.as(DraftCancelEventContext.class).getResult());
        case DraftService.EVENT_DRAFT_CREATE:
          return convert(context.as(DraftCreateEventContext.class).getResult());
        case DraftService.EVENT_DRAFT_READ:
          return convert(context.as(DraftReadEventContext.class).getResult());
        case DraftService.EVENT_ACTIVE_READ:
          return convert(context.as(ActiveReadEventContext.class).getResult());
        default:
          return null;
      }
    } else if (argumentType != ArgumentType.RESULT) {
      // go to CQN in BEFORE / ON phase
      switch (context.getEvent()) {
        case CqnService.EVENT_CREATE:
          return convert(
              ResultUtils.convert(context.as(CdsCreateEventContext.class).getCqn().entries()));
        case CqnService.EVENT_UPDATE:
          return convert(
              ResultUtils.convert(
                  context
                      .as(CdsUpdateEventContext.class)
                      .getCqn()
                      .entries())); // might be only partial
        case CqnService.EVENT_UPSERT:
          return convert(
              ResultUtils.convert(context.as(CdsUpsertEventContext.class).getCqn().entries()));
        case DraftService.EVENT_DRAFT_NEW:
          return convert(
              ResultUtils.convert(context.as(DraftNewEventContext.class).getCqn().entries()));
        case DraftService.EVENT_DRAFT_PATCH:
          return convert(
              ResultUtils.convert(
                  context
                      .as(DraftPatchEventContext.class)
                      .getCqn()
                      .entries())); // might be only partial
        case DraftService.EVENT_DRAFT_CREATE:
          return convert(
              ResultUtils.convert(context.as(DraftCreateEventContext.class).getCqn().entries()));
        default:
          return null;
      }
    }
    return null;
  }

  private Object convert(Result result) {
    switch (argumentType) {
      case RESULT:
        return result;
      case CDS_RESULT:
        return CdsResult.of(result, pojoClass);
      case STREAM:
        return result.streamOf(pojoClass);
      case LIST:
        return result.listOf(pojoClass);
      case SINGLE:
        return result.single(pojoClass);
      default:
        throw new IllegalStateException();
    }
  }

  @Override
  public void verifyOrThrow(HandlerDescriptor descriptor) {
    Set<String> events = new HashSet<>();
    for (AnnotationDescriptor ad : descriptor.getAnnotations()) {
      events.addAll(Arrays.asList(ad.getEvents()));
    }

    for (String event : events) {
      verifyEventOrThrow(event, descriptor);
    }
  }

  private void verifyEventOrThrow(String event, HandlerDescriptor descriptor) {
    if (!allowedEvents.contains(event)) {
      throw new ErrorStatusException(
          CdsErrorStatuses.POJO_ARGUMENT_MISMATCH,
          event,
          genericType.getTypeName(),
          descriptor.getMethodName());
    }
  }

  @Override
  public String[] indicateEntities() {
    CdsName cdsName = pojoClass.getAnnotation(CdsName.class);
    if (cdsName != null) {
      return new String[] {cdsName.value()};
    }
    return new String[0];
  }
}
