/*
 * © 2020-2025 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.impl.messages;

import com.sap.cds.impl.parser.PathParser;
import com.sap.cds.ql.CQL;
import com.sap.cds.ql.ElementRef;
import com.sap.cds.ql.StructuredType;
import com.sap.cds.ql.cqn.CqnReference;
import com.sap.cds.ql.cqn.CqnReference.Segment;
import com.sap.cds.ql.cqn.Path;
import com.sap.cds.ql.cqn.ResolvedSegment;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.services.messages.MessageTarget;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.function.Function;

public class MessageTargetImpl implements MessageTarget {

  private final String parameter;
  private final CqnReference reference;

  private MessageTargetImpl(String parameter, CqnReference reference) {
    this.parameter = parameter;
    this.reference = reference;
  }

  public static MessageTarget create(Path path, CdsElement element) {
    List<Segment> segments = new ArrayList<>();

    Iterator<ResolvedSegment> iterator = path.iterator();
    if (iterator.hasNext()) {
      iterator.next(); // skip root entity
    }
    iterator.forEachRemaining(resolvedSegment -> segments.add(resolvedSegment.segment()));
    segments.addAll(PathParser.segments(element.getName()));
    CqnReference ref;
    if (element.getType().isAssociation()) {
      ref = CQL.to(segments).asRef();
    } else {
      ref = CQL.get(segments);
    }
    return new MessageTargetImpl(MessageTarget.PARAMETER_CQN, ref);
  }

  public static MessageTarget create(String target) {
    return new MessageTargetImpl(target, null);
  }

  public static MessageTarget create(String parameter, CqnReference reference) {
    return new MessageTargetImpl(parameter, reference);
  }

  public static MessageTarget create(String parameter, Function<StructuredType<?>, Object> path) {
    return create(parameter, CQL.entity("SKIP"), path);
  }

  @SuppressWarnings("unchecked")
  public static <E extends StructuredType<E>> MessageTarget create(
      String parameter, Class<E> type, Function<E, Object> path) {
    return create(parameter, CQL.entity(type), (Function<StructuredType<?>, Object>) path);
  }

  private static MessageTarget create(
      String parameter, StructuredType<?> rootElement, Function<StructuredType<?>, Object> path) {
    boolean structured = false;
    List<Segment> segments = new ArrayList<>();

    Object p = path.apply(rootElement);
    if (p instanceof StructuredType type) {
      structured = true;
      segments.addAll(type.asRef().segments());
    } else if (p instanceof ElementRef ref) {
      segments.addAll(ref.segments());
    } else {
      // invalid specification
      return null;
    }

    String entityName = rootElement.asRef().firstSegment();
    if (entityName.equals(segments.get(0).id())) {
      segments.remove(0);
    }

    if (structured) {
      return new MessageTargetImpl(parameter, CQL.to(segments).asRef());
    } else {
      return new MessageTargetImpl(parameter, CQL.get(segments));
    }
  }

  @Override
  public CqnReference getRef() {
    return reference;
  }

  @Override
  public String getParameter() {
    return parameter;
  }
}
