/*
 * © 2021-2025 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.impl.outbox.persistence;

import com.sap.cds.services.application.ApplicationLifecycleService;
import com.sap.cds.services.application.ApplicationPreparedEventContext;
import com.sap.cds.services.application.ApplicationStoppedEventContext;
import com.sap.cds.services.environment.CdsProperties.Outbox.Persistent.Scheduler;
import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.After;
import com.sap.cds.services.handler.annotations.On;
import com.sap.cds.services.handler.annotations.ServiceName;
import com.sap.cds.services.impl.outbox.persistence.collectors.TenantCache;
import com.sap.cds.services.impl.scheduler.TaskScheduler;
import com.sap.cds.services.impl.scheduler.TaskScheduler.Task;
import com.sap.cds.services.impl.scheduler.TaskScheduler.TaskSchedule;
import com.sap.cds.services.mt.TenantProviderService;
import com.sap.cds.services.runtime.CdsRuntime;
import java.util.List;
import java.util.concurrent.ThreadLocalRandom;

@ServiceName(ApplicationLifecycleService.DEFAULT_NAME)
public class PersistentOutboxInitializationHandler implements EventHandler {

  private final TenantCache tenantCache;
  private final TaskScheduler taskScheduler;

  public PersistentOutboxInitializationHandler(
      TenantCache tenantCache, TaskScheduler taskScheduler) {
    this.tenantCache = tenantCache;
    this.taskScheduler = taskScheduler;
  }

  @After
  protected void initOutboxCollectors(ApplicationPreparedEventContext context) {
    if (context
        .getCdsRuntime()
        .getEnvironment()
        .getCdsProperties()
        .getEnvironment()
        .getCommand()
        .isEnabled()) {
      return;
    }

    if (taskScheduler != null) {
      taskScheduler.start();
    } else {
      tenantCache.start(context.getCdsRuntime());
    }

    context
        .getCdsRuntime()
        .getServiceCatalog()
        .getServices(PersistentOutbox.class)
        .forEach(PersistentOutbox::init);

    Scheduler schedulerCfg =
        context
            .getCdsRuntime()
            .getEnvironment()
            .getCdsProperties()
            .getOutbox()
            .getPersistent()
            .getScheduler();
    if (taskScheduler != null && schedulerCfg.getAllTenantsTask().isEnabled()) {
      taskScheduler.scheduleTask(
          "OutboxTenantsTask",
          new AllTenantsTask(context.getCdsRuntime()),
          schedulerCfg.getAllTenantsTask().getExecutionStartDelay().toMillis());
    }
  }

  @On
  protected void stopOutboxCollectors(ApplicationStoppedEventContext context) {
    context
        .getCdsRuntime()
        .getServiceCatalog()
        .getServices(PersistentOutbox.class)
        .forEach(PersistentOutbox::stop);
    if (taskScheduler != null) {
      taskScheduler.shutdown();
    } else {
      tenantCache.stop();
    }
  }

  /** All tenants task is responsible for periodic planning of tenant task for all outboxes */
  private static class AllTenantsTask implements Task {

    private final TenantProviderService tenantService;
    private final List<PersistentOutbox> outboxServices;

    private final long allTenantsInterval;
    private final long allTenantsSpreadTime;

    public AllTenantsTask(CdsRuntime runtime) {
      this.tenantService =
          runtime
              .getServiceCatalog()
              .getService(TenantProviderService.class, TenantProviderService.DEFAULT_NAME);
      this.outboxServices =
          runtime.getServiceCatalog().getServices(PersistentOutbox.class).toList();

      Scheduler schedulerCfg =
          runtime.getEnvironment().getCdsProperties().getOutbox().getPersistent().getScheduler();

      this.allTenantsInterval = schedulerCfg.getAllTenantsTask().getInterval().toMillis();
      this.allTenantsSpreadTime =
          schedulerCfg.getAllTenantsTask().getExecutionSpreadTime().toMillis();
    }

    @Override
    public TaskSchedule run() {
      tenantService
          .readTenants()
          .forEach(
              tenant -> {
                outboxServices.forEach(
                    outbox -> {
                      long delay = ThreadLocalRandom.current().nextLong(allTenantsSpreadTime);
                      outbox.schedule(tenant, delay, true);
                    });
              });
      return new TaskSchedule(this, allTenantsInterval, true);
    }
  }
}
