/*
 * © 2024-2025 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.impl.utils;

import com.sap.cds.ql.cqn.Path;
import com.sap.cds.reflect.CdsDefinition;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.model.CdsAnnotations;

public class ValidatorErrorUtils {

  private ValidatorErrorUtils() {
    // empty
  }

  public static void handleValidationError(
      EventContext context,
      Path path,
      CdsElement element,
      CdsAnnotations messageAnnotation,
      CdsErrorStatuses status,
      Object... args) {
    handleValidationError(context, false, path, element, messageAnnotation, status, args);
  }

  @SuppressWarnings("deprecation")
  public static void handleValidationError(
      EventContext context,
      boolean forceException,
      Path path,
      CdsElement element,
      CdsAnnotations messageAnnotation,
      CdsErrorStatuses status,
      Object... args) {
    // validations on action/function parameters are considered transition messages
    boolean isTransition =
        CdsModelUtils.isInternalOperationType(element.getDeclaringType().as(CdsDefinition.class));
    if (!forceException
        && context.getCdsRuntime().getEnvironment().getCdsProperties().getErrors().isCombined()) {
      context
          .getMessages()
          .error(getMessageKey(messageAnnotation, element, status), args)
          .code(status.getCodeString())
          .target(path, element)
          .transition(isTransition);
    } else {
      throw new ErrorStatusException(status, args)
          .messageTarget(path, element)
          .transition(isTransition);
    }
  }

  public static String getMessageKey(
      CdsAnnotations messageAnnotation, CdsElement element, CdsErrorStatuses status) {
    String messageKey = status.getCodeString();
    if (messageAnnotation != null) {
      String customMessage = messageAnnotation.getOrDefault(element);
      if (customMessage != null) {
        if (customMessage.startsWith("{i18n>")) {
          int end = customMessage.lastIndexOf('}');
          if (end >= 6) {
            messageKey = customMessage.substring(6, end);
          }
        } else {
          messageKey = customMessage;
        }
      }
    }
    return messageKey;
  }
}
