/*
 * © 2022-2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.impl.application;

import com.sap.cds.services.ErrorStatus;
import com.sap.cds.services.ErrorStatuses;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.ServiceException;
import com.sap.cds.services.application.ApplicationLifecycleService;
import com.sap.cds.services.application.ErrorResponseEventContext;
import com.sap.cds.services.application.ErrorResponseEventContext.ErrorResponse;
import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.HandlerOrder;
import com.sap.cds.services.handler.annotations.On;
import com.sap.cds.services.handler.annotations.ServiceName;
import com.sap.cds.services.messages.Message;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.OrderConstants;
import java.util.ArrayList;
import java.util.List;

/**
 * This handler is called when @{@link ApplicationLifecycleService} handles the event {@link
 * ApplicationLifecycleService#EVENT_ERROR_RESPONSE} On @On phase, it merges the messages from the
 * exception and request context in error response that can be changed by custom @After handler.
 */
@ServiceName(value = "*", type = ApplicationLifecycleService.class)
public class ErrorResponseHandler implements EventHandler {

  @On
  @HandlerOrder(OrderConstants.On.DEFAULT_ON)
  public void onErrorResponse(ErrorResponseEventContext context) {
    ServiceException exception = context.getException();

    ErrorResponse response = ErrorResponse.create();
    response.setHttpStatus(exception.getErrorStatus().getHttpStatus());

    List<Message> messages = new ArrayList<>();
    messages.add(toMessage(exception, context));
    context.getMessages().stream().forEach(messages::add);
    response.setMessages(messages);

    context.setResult(response);
  }

  private Message toMessage(ServiceException exception, EventContext context) {
    String localizedMessage = exception.getLocalizedMessage();
    String text = localizedMessage == null ? getDefaultDescription() : localizedMessage;
    Message message = Message.create(text, exception);

    ErrorStatus errorStatus = exception.getErrorStatus();
    boolean stackMessagesEnabled =
        context
            .getCdsRuntime()
            .getEnvironment()
            .getCdsProperties()
            .getErrors()
            .getStackMessages()
            .isEnabled();
    // TODO should we display the correct CAP internal code here, in case a translation in app
    // resources was found?
    if (errorStatus instanceof CdsErrorStatuses && !stackMessagesEnabled) {
      message.code(String.valueOf(errorStatus.getHttpStatus()));
    }
    return message;
  }

  private String getDefaultDescription() {
    return new ErrorStatusException(ErrorStatuses.SERVER_ERROR).getLocalizedMessage();
  }
}
