/*
 * © 2021-2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.impl.auditlog;

import com.sap.cds.Result;
import com.sap.cds.impl.DataProcessor;
import com.sap.cds.ql.CQL;
import com.sap.cds.ql.cqn.CqnSelect;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.services.auditlog.Access;
import com.sap.cds.services.auditlog.AuditLogService;
import com.sap.cds.services.cds.ApplicationService;
import com.sap.cds.services.cds.CdsReadEventContext;
import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.After;
import com.sap.cds.services.handler.annotations.Before;
import com.sap.cds.services.handler.annotations.HandlerOrder;
import com.sap.cds.services.handler.annotations.ServiceName;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.OrderConstants;
import com.sap.cds.services.utils.model.CdsAnnotations;
import com.sap.cds.util.CqnStatementUtils;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;

/**
 * This handler is registered on READ events of the {@link ApplicationService}. It detects accesses
 * and modifications of personal data and logs them to the {@link AuditLogService}. Entity fields
 * can be annotated with {@link CdsAnnotations#PERSONALDATA_POTENTIALLYPERSONAL}, {@link
 * CdsAnnotations#PERSONALDATA_POTENTIALLYSENSITIVE} to mark them as personal data.
 *
 * @see <a
 *     href="https://pages.github.tools.sap/cap/docs/guides/data-privacy#indicate-privacy">Indicate
 *     Personal Data In Your Domain Model</a>
 * @see <a href="https://pages.github.tools.sap/cap/docs/guides/data-privacy#audit-log">Audit
 *     Logging</a>
 */
@ServiceName(value = "*", type = ApplicationService.class)
class ApplicationServicePersonalDataHandler implements EventHandler {
  private static final String PD_ANALYZER_ACCESS_KEY =
      PersonalDataAnalyzerAccess.class.getCanonicalName();

  private final PersonalDataCaches caches;
  private final CdsRuntime runtime;
  private final AuditLogService auditLog;

  ApplicationServicePersonalDataHandler(PersonalDataCaches caches, CdsRuntime runtime) {
    this.caches = Objects.requireNonNull(caches, "caches must not be null");
    this.runtime = Objects.requireNonNull(runtime, "runtime must not be null");
    this.auditLog =
        runtime.getServiceCatalog().getService(AuditLogService.class, AuditLogService.DEFAULT_NAME);
  }

  @Before
  @HandlerOrder(OrderConstants.Before.ADAPT_STATEMENT)
  protected void beforeRead(CdsReadEventContext context) {
    CdsEntity entity = context.getTarget();

    // use datamodifier to select datasubject Ids and keys of data object
    PersonalDataModifier modifier = new PersonalDataModifier(entity, false, this.caches);
    CqnSelect cqnSelect = CQL.copy(modifySelectStatement(entity, context.getCqn()), modifier);

    PersonalDataAnalyzerAccess analyzer =
        new PersonalDataAnalyzerAccess(
            entity, modifier.getSensitiveElements(), this.caches, this.runtime);

    // check if target entity has personal data annotations
    if (analyzer.hasPersonalData()) {
      context.setCqn(cqnSelect);

      context.put(PD_ANALYZER_ACCESS_KEY, analyzer);
    }
  }

  @After
  @HandlerOrder(OrderConstants.After.AUDIT)
  protected void afterRead(CdsReadEventContext context) {
    PersonalDataAnalyzerAccess pdAnalyzer =
        (PersonalDataAnalyzerAccess) context.get(PD_ANALYZER_ACCESS_KEY);

    if (pdAnalyzer != null) {
      context.put(PD_ANALYZER_ACCESS_KEY, null);

      Result result = context.getResult();

      // store read data in analyzer, required to create data access objects
      pdAnalyzer.setData(context.getResult().list());

      List<Access> accesses = pdAnalyzer.getDataAccesses();
      if (!accesses.isEmpty()) {
        this.auditLog.logDataAccess(accesses);
      }

      DataProcessor.create()
          .action(
              (type, entry) ->
                  entry
                      .keySet()
                      .removeIf(k -> k.startsWith(PersonalDataUtils.PERSONAL_DATA_PREFIX)))
          .process(result);
      // TODO switch to aliases for expands as soon as it's available in CDS4J
      removeEmptyMaps(result);
    }
  }

  private static CqnSelect modifySelectStatement(CdsEntity entity, CqnSelect select) {
    CqnSelect newSelect = CqnStatementUtils.resolveKeyPlaceholder(entity, select);
    newSelect = CqnStatementUtils.resolveStar(newSelect, entity);
    return newSelect;
  }

  @SuppressWarnings("unchecked")
  private static void removeEmptyMaps(Iterable<? extends Map<String, Object>> result) {
    result.forEach(
        row -> {
          Iterator<Entry<String, Object>> iter = row.entrySet().iterator();
          while (iter.hasNext()) {
            Entry<String, Object> entry = iter.next();
            if (entry.getValue() instanceof Map) {
              removeEmptyMaps(Arrays.asList((Map<String, Object>) entry.getValue()));
              if (((Map<?, ?>) entry.getValue()).isEmpty()) {
                iter.remove();
              }
            } else if (entry.getValue() instanceof List) {
              List<?> valueList = (List<?>) entry.getValue();
              if (!valueList.isEmpty() && valueList.get(0) instanceof Map) {
                removeEmptyMaps((List<Map<String, Object>>) entry.getValue());
              }
            }
          }
        });
  }
}
